<?php

namespace Ignite\Core\Models\Report;

use Ignite\Core\Entities\Filters\QueryPermissionFilters;
use Ignite\Core\Entities\User;
use Ignite\Core\Http\Forms\ProfileForm;
use Ignite\Core\Models\Grid\ParticipantFormatter;
use Ignite\Core\Models\Grid\QueryTable;
use Ignite\Core\Traits\ReportFormConfiguration;
use Illuminate\Database\DatabaseManager;
use Illuminate\Support\Collection;
use Yajra\DataTables\DataTables;

class EnrollmentReport extends QueryTable
{
    use ReportFormConfiguration {
        ReportFormConfiguration::getColumns as getFormColumns;
    }

    /**
     * The columns we should always display in the report.
     * @var array
     */
    protected $alwaysDisplay = [];

    /**
     * The form to configure report columns when using ReportFormConfiguration trait.
     * @var string
     */
    protected $form = ProfileForm::class;

    /**
     * @var ParticipantFormatter
     */
    protected $formatter;

    /**
     * EnrollmentReport constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param ParticipantFormatter $formatter
     * @param array $params
     */
    public function __construct(
        DataTables           $datatables,
        DatabaseManager      $databaseManager,
        ParticipantFormatter $formatter,
        array                $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);
        $this->formatter = $formatter;
    }

    /**
     * @return array
     */
    protected function getColumns(): array
    {
        $columns = $this->getFormColumns();
        unset($columns['ssn']);

        return $columns;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns(): array
    {
        return [
            'actions' => $this->actionsColumn(),
            'user_id' => [
                'title' => 'User ID',
                'name' => 'participant.user_id',
                'visible' => true,
                'orderable' => true,
                'exportable' => true,
            ],
            'created_at' => [
                'title' => 'Enrollment Date',
                'name' => 'participant.created_at',
                'visible' => true,
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Query\Builder|Collection
     */
    public function query(): \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Query\Builder|Collection
    {
        $columns = $this->getColumnNames(['actions'])->toArray();

        $query = $this->getConnection()
            ->table('core_participant as participant')
            ->select($columns)
            ->leftJoin('core_user as user', 'user.user_id', '=', 'participant.user_id')
            ->where('user.internal', User::TYPE_PARTICIPANT)
            ->where('user.status', User::STAT_ACTIVE);

        return QueryPermissionFilters::for('core.user.participant.browse')
            ->apply($query, 'participant');
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'agree_confirm_1' => function ($model) { return $this->formatter->agreeConfirm1($model); },
            'agree_confirm_2' => function ($model) { return $this->formatter->agreeConfirm2($model); },
            'archived' => function ($model) { return $this->formatter->archived($model); },
            'internal' => function ($model) { return $this->formatter->internal($model); },
            'status' => function ($model) { return $this->formatter->status($model); },
        ];
    }
}
