<?php

namespace Ignite\Core\Http\Controllers\Admin;

use Ignite\Core\Contracts\Repositories\BlockRepository;
use Ignite\Core\Entities\Block;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Http\Requests\CreateBlockRequest;
use Ignite\Core\Http\Requests\UpdateBlockRequest;
use Ignite\Core\Http\Requests\DestroyBlockRequest;
use Ignite\Core\Http\Requests\DestroyManyBlocksRequest;
use Ignite\Core\Models\Grid\BlockTable;

class BlockController extends Controller
{
    /** @var BlockRepository */
    protected $blockRepository;

    /**
     * Create a new controller instance.
     *
     * @param BlockRepository $blockRepository
     */
    public function __construct(BlockRepository $blockRepository)
    {
        $this->blockRepository = $blockRepository;
    }

    /**
     * Display a listing of the pages resource.
     *
     * @param  BlockTable $table
     * @return \Illuminate\Http\JsonResponse|\Illuminate\View\View
     */
    public function index(BlockTable $table)
    {
        if (request()->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Core::admin.cms.blocks.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $block = new Block();

        return view('Core::admin.cms.blocks.create', compact('block'));
    }

    /**
     * Store a newly created resource in storage.
     * @param  CreateBlockRequest $request
     * @return \Illuminate\Http\Response
     */
    public function store(CreateBlockRequest $request)
    {
        try {
            $this->blockRepository->create($request->all());
            $this->flash('success', 'Block updated successfully.');
        } catch (\Exception $e) {
            $this->flash('error', $e->getMessage());
        }

        return redirect()->route('admin.blocks.index');
    }

    /**
     * Preview the specified page.
     *
     * @return \Illuminate\Http\Response
     *
    public function show()
    {
        return view('Cms::admin.pages.show');
    }*/

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $block = $this->blockRepository->find($id);

        return view('Core::admin.cms.blocks.edit', compact('block', 'id'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  UpdateBlockRequest $request
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateBlockRequest $request, $id)
    {
        try {
            $this->blockRepository->update($id, $request->all());
            $this->flash('success', 'Block updated successfully.');
        } catch (\Exception $e) {
            $this->flash('error', $e->getMessage());
        }

        return redirect()->route('admin.blocks.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @param  DestroyBlockRequest $request
     * @return \Illuminate\Http\Response
     */
    public function destroy($id, DestroyBlockRequest $request)
    {
        try {
            $this->blockRepository->delete($id);
            $message = "Block with ID: $id was deleted successfully.";
            $status = 'success';
        } catch (\Exception $e) {
            $message = "Block with ID: $id could not be deleted. Error: " . $e->getMessage();
            $status = 'error';
        }

        if ($request->wantsJson()) {
            return response()->json(['status' => $status, 'message' => $message]);
        }

        $this->flash($status, $message);

        return redirect()->back();
    }

    /**
     * Destroy many blocks using the provided IDs.
     *
     * @param  DestroyManyBlocksRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyMany(DestroyManyBlocksRequest $request)
    {
        try {
            $count = $this->blockRepository->deleteMany(explode(',', $request->ids));
            $this->flash('success', $count . " blocks were deleted successfully.");
        } catch (\Exception $e) {
            $this->flash('error', $e->getMessage());
        }

        return redirect()->route('admin.blocks.index');
    }
}
