<?php

namespace Ignite\Core\Tests\Unit\Models\Import;

use Ignite\Core\Entities\Import;
use Ignite\Core\Models\Import\LogFormatter;
use Ignite\Core\Tests\TestCase;
use Illuminate\Config\Repository;

class LogFormatterTest extends TestCase
{
    private function getLogFormatter(array $lines): LogFormatter
    {
        $configRepository = app(Repository::class);
        /** @var Import $import */
        $import = factory(Import::class)->make();

        $import->disk()->delete($import->log);
        foreach ($lines as $line) {
            $import->disk()->append($import->log, $line);
        }

        return new LogFormatter($import, $configRepository);
    }

    /**
     * @dataProvider formats_log_messages_provider
     * @test
     */
    public function formats_log_messages(array $logLines, array $expectedFormatResult)
    {
        // Arrange
        $logFormatter = $this->getLogFormatter($logLines);

        // Execute
        $formattedLines = $logFormatter->format();

        // Check
        $this->assertEquals($expectedFormatResult, $formattedLines->toArray());
    }

    public function formats_log_messages_provider()
    {
        // Simple
        yield [
            [
                "[2021-11-09T19:59:38.176347-06:00] import.ERROR: Some message here [] []",
                "[2021-11-09T19:59:38.176347-06:00] import.INFO: Some other message here [] []",
            ],
            [
                [
                    "date" => "2021-11-09 07:59:38",
                    "icon" => "exclamation-circle",
                    "type" => "Error",
                    "color" => "text-danger",
                    "message" => "Some message here",
                    "context" => [],
                ],
                [
                    "date" => "2021-11-09 07:59:38",
                    "icon" => "check",
                    "type" => "Success",
                    "color" => "text-green",
                    "message" => "Some other message here",
                    "context" => [],
                ],
            ],
        ];

        // Message has square bracket
        yield [
            [
                "[2021-11-09T19:59:38.176347-06:00] import.ERROR: Rejected line `354` => SQLSTATE[HY000]: General error: 1364 Field 'activity_id' doesn't have a default value [] []",
                "[2021-11-09T19:59:38.176347-06:00] import.ERROR: Rejected line `354` => SQLSTATEHY000: General error: 1364 'activity_id' doesn't [] []",
            ],
            [
                [
                    "date" => "2021-11-09 07:59:38",
                    "icon" => "exclamation-circle",
                    "type" => "Error",
                    "color" => "text-danger",
                    "message" => "Rejected line `354` => SQLSTATE[HY000]: General error: 1364 Field 'activity_id' doesn't have a default value",
                    "context" => [],
                ],
                [
                    "date" => "2021-11-09 07:59:38",
                    "icon" => "exclamation-circle",
                    "type" => "Error",
                    "color" => "text-danger",
                    "message" => "Rejected line `354` => SQLSTATEHY000: General error: 1364 'activity_id' doesn't",
                    "context" => [],
                ],
            ],
        ];

        // Line has context
        yield [
            [
                '[2021-10-15T18:27:12.297517+00:00] import.INFO: Imported claim on line `168` {"Claim ID":"3869","client_claim_id":"19340","user_id":362,"prm_number":"259288","distributor_po_number":"172049","status":"approved","value":"247.80","offer_promotion_id":"1","email":"kyle.rosenberg@rjyoung.com","id":3869} []',
                '[2021-10-15T18:27:12.297517+00:00] import.ERROR: Some message with json {"test": "11"} {"Claim ID":"3869","client_claim_id":"19340","user_id":362,"prm_number":"259288","distributor_po_number":"172049","status":"approved","value":"247.80","offer_promotion_id":"1","email":"kyle.rosenberg@rjyoung.com","id":3869} []',
            ],
            [
                [
                    'date' => '2021-10-15 06:27:12',
                    'icon' => 'check',
                    'type' => 'Success',
                    'color' => 'text-green',
                    'message' => 'Imported claim on line `168`',
                    'context' => [
                        'Claim ID' => '3869',
                        'client_claim_id' => '19340',
                        'user_id' => 362,
                        'prm_number' => '259288',
                        'distributor_po_number' => '172049',
                        'status' => 'approved',
                        'value' => '247.80',
                        'offer_promotion_id' => '1',
                        'email' => 'kyle.rosenberg@rjyoung.com',
                        'id' => 3869,
                    ],
                ],
                [
                    'date' => '2021-10-15 06:27:12',
                    "icon" => "exclamation-circle",
                    "type" => "Error",
                    "color" => "text-danger",
                    'message' => 'Some message with json {"test": "11"}',
                    'context' => [
                        'Claim ID' => '3869',
                        'client_claim_id' => '19340',
                        'user_id' => 362,
                        'prm_number' => '259288',
                        'distributor_po_number' => '172049',
                        'status' => 'approved',
                        'value' => '247.80',
                        'offer_promotion_id' => '1',
                        'email' => 'kyle.rosenberg@rjyoung.com',
                        'id' => 3869,
                    ],
                ],
            ],
        ];
    }
}
