<?php

namespace Ignite\Core\Tests\Unit\Files\Processors;

use Ignite\Core\Files\File;
use Ignite\Core\Files\Processors\MoveFileProcessor;
use Ignite\Core\Tests\TestCase;
use Illuminate\Support\Facades\Storage;

class MoveFileProcessorTest extends TestCase
{
    /** @test */
    public function moves_a_file_to_a_new_directory_on_the_same_disk()
    {
        // Arrange
        $sftp = Storage::fake('an-sftp');
        $file = new File('dir1/my-file.txt', 'an-sftp');
        $file->put('test');

        // Pre-check
        $this->assertCount(1, $sftp->files('dir1'));
        $this->assertCount(0, $sftp->files('dir2'));

        // Execute
        $newFile = (new MoveFileProcessor('dir2'))->run($file);

        // Arrange
        $this->assertFalse($file->is($newFile));
        $this->assertEquals($file->getBasename(), $newFile->getBasename());
        $file->assertMissing();
        $newFile->assertExists();
        $this->assertEquals('test', $newFile->get());
        $this->assertCount(0, $sftp->files('dir1'));
        $this->assertCount(1, $sftp->files('dir2'));
    }

    /** @test */
    public function moves_a_file_to_a_new_directory_on_a_new_disk()
    {
        // Arrange
        $disk1 = Storage::fake('an-sftp');
        $disk2 = Storage::fake('another-disk');
        $file = new File('dir1/my-file.txt', 'an-sftp');
        $file->put('test');

        // Pre-check
        $this->assertCount(1, $disk1->files('dir1'));
        $this->assertCount(0, $disk2->files('dir1'));

        // Execute
        $newFile = (new MoveFileProcessor('dir1', 'another-disk'))->run($file);

        // Arrange
        $this->assertFalse($file->is($newFile));
        $this->assertEquals($file->getBasename(), $newFile->getBasename());
        $file->assertMissing();
        $newFile->assertExists();
        $this->assertEquals('test', $newFile->get());
        $this->assertCount(0, $disk1->files('dir1'));
        $this->assertCount(1, $disk2->files('dir1'));
    }

    /** @test */
    public function moves_a_file_to_a_new_directory_on_a_new_disk_and_gives_it_a_new_name()
    {
        // Arrange
        $disk1 = Storage::fake('an-sftp');
        $disk2 = Storage::fake('another-disk');
        $file = new File('dir1/my-file.txt', 'an-sftp');
        $file->put('test');

        // Pre-check
        $this->assertCount(1, $disk1->files('dir1'));
        $this->assertCount(0, $disk2->files('dir1'));

        // Execute
        $newFile = (new MoveFileProcessor('dir1', 'another-disk'))
            ->withName('another-name.txt')
            ->run($file);

        // Arrange
        $this->assertFalse($file->is($newFile));
        $this->assertEquals('another-name.txt', $newFile->getBasename());
        $file->assertMissing();
        $newFile->assertExists();
        $this->assertEquals('test', $newFile->get());
        $this->assertCount(0, $disk1->files('dir1'));
        $this->assertCount(1, $disk2->files('dir1'));
    }
}
