<?php

namespace Ignite\Core\Models\Menu;

use Ignite\Core\Contracts\CanDisplayInMenu;
use Ignite\Core\Events\Menus\ParticipantMenuPreCompile;
use Illuminate\Support\Str;
use Spatie\Menu\Laravel\Facades\Menu;
use Spatie\Menu\Laravel\Html;
use Spatie\Menu\Laravel\Link;
use Spatie\Menu\Laravel\Menu as SpatieMenu;

class ParticipantMenu extends Manager
{
    /** @var SpatieMenu */
    private $menu;

    /**
     * Add a link to the menu.
     *
     * @param CanDisplayInMenu $item
     * @param SpatieMenu $menu
     */
    protected function addLink(CanDisplayInMenu|array $item, $menu)
    {
        $link = $menu->link($item->getUrl(), $this->buildLabel($item));
        $link->setActive(function ($link) use ($item) {
            if ($link->url() === url()->current()) {
                return true;
            }

            if (Str::contains(url()->current(), $link->url()) && url($link->url()) !== url('admin')) {
                return true;
            }

            return false;
        });
    }

    /**
     * Build the label for the link,
     *
     * @param  CanDisplayInMenu $item
     * @return string
     */
    protected function buildLabel(CanDisplayInMenu $item)
    {
        return Html::raw(sprintf(
            '<i class="fa-solid fa-%s"></i><span>%s</span><span class="label label-default pull-right">%s</span>',
            $item->getIcon(),
            $item->getLabel(),
            $item->getBadge()
        ))->html();
    }

    /**
     * Build a submenu link.
     *
     * @param  CanDisplayInMenu $item
     * @return Link
     */
    protected function addSubmenuLink(CanDisplayInMenu $item)
    {
        $child = sprintf(
            '<i class="fa-solid fa-fw fa-%s"></i><span>%s</span> <span class="pull-right-container"><i class="fa-solid fa-angle-left pull-right"></i></span>',
            $item->getIcon(),
            $item->getLabel()
        );

        return Link::to('#', $child);
    }

    /**
     * Build the menu.
     *
     * @param  \Illuminate\Support\Collection|array $links
     * @return SpatieMenu
     */
    public function buildMenu($links): SpatieMenu
    {
        return Menu::build($links->toArray(), function ($menu, $item) {
            if (is_array($item)) {
                $item = new Item($item['url'], $item['label'], $item['position'], $item['allow'], $item['children'], $item['icon'], $item['badge']);
            }

            if (! $item->hasChildren()) {
                $this->addLink($item, $menu);
            } else {
                $menu->submenu(
                    $this->addSubmenuLink($item),
                    $this->buildMenu($item->getChildren())
                         ->addClass('treeview-menu')
                         ->addParentClass('treeview')
                );
            }
        });
    }

    /**
     * Render the menu.
     *
     * @param  int $participantId
     * @return string
     */
    public function render(int $participantId)
    {
        event(new ParticipantMenuPreCompile($this, $participantId));

        $this->menu = $this->buildMenu($this->getItems())->addClass('nav nav-pills nav-stacked');

        return $this->menu->render();
    }
}
