<?php

namespace Ignite\Core\Models\Grid;

use Ignite\Core\Entities\User;
use Illuminate\Support\HtmlString;

class UserFormatter extends Formatter
{
    /**
     * Custom column for actions.
     *
     * @param  \Ignite\Core\Entities\User $model
     * @return HtmlString
     */
    public function actions(object $model): HtmlString|string
    {
        $actions = [];

        if (auth()->user()->can('core.security.users.update')) {
            $actions[] = [
                'link' => route('admin.security.users.edit', $model->user_id),
                'label' => 'Assign Groups',
                'icon' => 'random'
            ];
        }

        if (auth()->user()->can('core.security.users.activate') && ($model->status === User::STAT_INACTIVE || $model->status === User::STAT_PENDING)) {
            $actions[] = [
                'link' => route('admin.security.users.activate', $model->user_id),
                'label' => 'Activate',
                'icon' => 'check',
                'text' => 'text-success'
            ];
        }

        if (auth()->user()->can('core.security.users.deactivate') && $model->status === User::STAT_ACTIVE) {
            $actions[] = [
                'link' => route('admin.security.users.deactivate', $model->user_id),
                'label' => 'Deactivate',
                'icon' => 'times',
                'text' => 'text-danger'
            ];
        }

        if (auth()->user()->can('core.user.impersonate')) {
            $actions[] = [
                'link' => route('admin.impersonate', $model->user_id),
                'label' => 'Impersonate',
                'icon' => 'user-secret'
            ];
        }

        return $this->html($this->buttons($actions));
    }

    /**
     * Wrap the group name with a link to the group.
     *
     * @param object $model
     * @return HtmlString
     */
    public function group(object $model): HtmlString|string
    {
        return $model->group_name;
    }

    /**
     * Wrap the first name with a link to the user.
     *
     * @param object $model
     * @return HtmlString
     */
    public function first(object $model): HtmlString|string
    {
        return $this->html(sprintf('<a href="%s">%s</a>', route('admin.participant.show', $model->user_id), $model->first));
    }

    /**
     * Wrap the last name with a link to the user.
     *
     * @param object $model
     * @return HtmlString
     */
    public function last(object $model): HtmlString|string
    {
        return $this->html(sprintf('<a href="%s">%s</a>', route('admin.participant.show', $model->user_id), $model->last));
    }

    /**
     * Format the participant's full name.
     *
     * @param  object $model
     * @return string
     */
    public function name(object $model): HtmlString|string
    {
        return $this->html(sprintf(
            '<a href="%s">%s %s</a>',
            route('admin.participant.show', $model->user_id),
            $model->first,
            $model->last
        ));
    }

    /**
     * Wrap the email with a link to the user.
     *
     * @param object $model
     * @return HtmlString
     */
    public function email(object $model): HtmlString|string
    {
        return $this->html(sprintf('<a href="%s">%s</a>', route('admin.participant.show', $model->user_id), $model->email));
    }

    /**
     * Format type of user.
     *
     * @param  object $model
     * @return string
     * @throws \Exception
     */
    public function type(object $model): HtmlString|string
    {
        $types = User::getTypeList();

        $type = $types[$model->internal];

        return $this->html($type ?? 'Unknown');
    }

    /**
     * Format status of the user.
     *
     * @param  object  $model
     * @return HtmlString
     * @throws \Exception
     */
    public function status(object $model): HtmlString|string
    {
        $types = User::getStatusList();

        $type = $types[$model->status];

        return $this->html($type ?? 'Unknown');
    }

    /**
     * Format archived as Yes or No.
     *
     * @param  object $model
     * @return string
     */
    public function archived(object $model): HtmlString|string
    {
        return $this->formatYesNo($model->archived);
    }

    /**
     * Get a participant checkbox based on their ID.
     *
     * @param  object $model
     * @return HtmlString
     */
    public function checkbox($model): HtmlString|string
    {
        return $this->formatCheckbox($model->user_id);
    }

    /**
     * Format the user created at date.
     *
     * @param  object  $model
     * @return HtmlString
     * @throws \Exception
     */
    public function createdAt(object $model): HtmlString
    {
        return $this->formatDate($model, 'created_at', 'Y-m-d H:i:s');
    }

    /**
     * Format the user updated at date.
     *
     * @param  object  $model
     * @return HtmlString
     * @throws \Exception
     */
    public function updatedAt(object $model): HtmlString
    {
        return $this->formatDate($model, 'updated_at', 'Y-m-d H:i:s');
    }

    /**
     * Format the user updated at date.
     *
     * @param  object $model
     * @return HtmlString
     * @throws \Exception
     */
    public function lastLoginAt(object $model): HtmlString|string
    {
        return $this->formatDate($model, 'last_login_at', 'Y-m-d H:i:s');
    }
}
