<?php

namespace Ignite\Core\Models\Grid;

use Ignite\Core\Entities\Filters\QueryPermissionFilters;
use Illuminate\Database\DatabaseManager;
use Yajra\DataTables\DataTables;

class ParticipantTransactionTable extends QueryTable
{
    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [
        'internal_notes', 'idc',
    ];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [
        'user_id', 'value',
    ];

    /**
     * @var ParticipantFormatter
     */
    protected $formatter;

    /**
     * ParticipantTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param TransactionFormatter $formatter
     * @param array $params
     */
    public function __construct(
        DataTables           $datatables,
        DatabaseManager      $databaseManager,
        TransactionFormatter $formatter,
        array                $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;

        if ((bool)config('core.use_email_as_username')) {
            $this->excludedKeys[] = 'username';
        }
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'user_id' => [
                'title' => 'User ID',
                'name' => 'transaction.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'type' => [
                'title' => 'Type',
                'name' => 'transaction.type',
                'orderable' => true,
                'exportable' => true,
            ],
            'related_name' => [
                'title' => 'Name',
                'name' => 'transaction.related_name',
                'orderable' => true,
                'exportable' => true,
            ],
            'description' => [
                'title' => 'Description',
                'name' => 'transaction.description',
                'orderable' => true,
                'exportable' => true,
            ],
            'notes' => [
                'title' => 'Notes',
                'name' => 'transaction.notes',
                'orderable' => true,
                'exportable' => true,
            ],
            'transaction_date' => [
                'title' => 'Date',
                'name' => 'transaction.transaction_date',
                'orderable' => true,
                'exportable' => true,
            ],
            'value' => [
                'title' => 'Value',
                'name' => 'transaction.value',
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The participant transactions query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = collect($this->getColumns())
            ->keyBy('name')
            ->except('actions')
            ->keys()
            ->toArray();

        $query = $this->getConnection()
            ->table('core_transaction as transaction')
            ->select($columns)
            ->leftJoin('core_user', 'core_user.user_id', '=', 'transaction.user_id')
            ->where('transaction.user_id', $this->user_id);

        return QueryPermissionFilters::for('core.user.transaction.browse')
            ->apply($query, 'transaction');
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'action' => [$this->formatter, 'actions'],
            'value' => [$this->formatter, 'balance'],
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters()
    {
        return parent::getBuilderParameters([
            'order' => [
                [5, 'desc'] // Transaction Date
            ],
        ]);
    }
}
