<?php

namespace Ignite\Core\Http\Controllers;

use Exception;
use Ignite\Core\Contracts\Repositories\PageRepository;
use Ignite\Core\Entities\Page;
use Ignite\Flash\Facades\Flash;
use Ignite\Flash\MessageHandler;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Kris\LaravelFormBuilder\FormBuilderTrait;

class Controller extends BaseController
{
    use AuthorizesRequests, DispatchesJobs, ValidatesRequests, FormBuilderTrait;

    /**
     * Log the exception.
     *
     * @param Exception $exception
     */
    protected function logException(Exception $exception)
    {
        logger()->error($exception->getMessage(), [
            'file' => $exception->getFile(),
            'code' => $exception->getCode(),
            'line' => $exception->getLine(),
            'class' => get_class($exception),
            'trace' => $exception->getTraceAsString()
        ]);
    }

    /**
     * Get the registration page content from the database.
     *
     * @param  string $key
     * @return Page
     */
    protected function getPageContent($key)
    {
        $pageRepository = app(PageRepository::class);
        $page = $pageRepository->findByCode($key);

        if (! $page) {
            $page = app(Page::class);
        }

        return $page;
    }

    /**
     * Flash a translated success message to the session.
     *
     * @param  string  $message
     * @param  array  $data
     * @return MessageHandler
     */
    protected function flashSuccess($message, $data = [])
    {
        return $this->flashMessage('success', $message, $data);
    }

    /**
     * Flash a translated error message to the session.
     *
     * @param  string  $message
     * @param  array  $data
     * @return MessageHandler
     */
    protected function flashError($message, $data = [])
    {
        return $this->flashMessage('error', $message, $data);
    }

    /**
     * Flash a translated warning message to the session.
     *
     * @param  string  $message
     * @param  array  $data
     * @return MessageHandler
     */
    protected function flashWarning($message, $data = [])
    {
        return $this->flashMessage('warning', $message, $data);
    }

    /**
     * Flash a translated info message to the session.
     *
     * @param  string  $message
     * @param  array  $data
     * @return MessageHandler
     */
    protected function flashInfo($message, $data = [])
    {
        return $this->flashMessage('info', $message, $data);
    }

    /**
     * Flash a translated message to the session.
     *
     * @param  string  $type
     * @param  string  $message
     * @param  array  $data
     * @return MessageHandler
     */
    protected function flashMessage($type, $message, $data = [])
    {
        return $this->flash($type, trans($message, $data));
    }

    /**
     * Flash a message to the session.
     *
     * @param  string  $type
     * @param  string  $message
     * @return MessageHandler|null
     */
    protected function flash($type, $message)
    {
        if (is_array($message)) {
            foreach ($message as $item) {
                Flash::add($item, $type);
            }

            return null;
        }

        return Flash::add($message, $type);
    }
}
