<?php

namespace Ignite\Core\Http\ViewComposers;

use Illuminate\Support\Carbon;
use Illuminate\View\View;

class DashboardDateRangePickerComposer
{
    /**
     * Bind data to the view.
     *
     * @param  View $view
     * @return void
     */
    public function compose(View $view)
    {
        $view->with('types', $this->types());
        $view->with('months', $this->months());
        $view->with('quarters', $this->quarters());
        $view->with('years', $this->years());
        $view->with('selectedType', $type = $this->selectedType());
        $view->with('selectedCurrent', $this->selectedCurrentStart($type));
        $view->with('selectedPrevious', $this->selectedPreviousStart($type));
    }

    /**
     * The types of comparison options.
     *
     * @return string[]
     */
    protected function types()
    {
        return [
            'range' => 'Range',
            'month' => 'Monthly',
            'quarter' => 'Quarterly',
            'year' => 'Yearly',
        ];
    }

    /**
     * The available months from which we can choose.
     *
     * @param int $max
     *
     * @return array
     */
    protected function months($max = 12)
    {
        return array_combine(range(1, $max), array_map(function ($month) {
            return Carbon::createFromDate(today()->year, $month, today()->day)->monthName;
        }, range(1, $max)));
    }

    /**
     * The available quarters from which we can choose.
     *
     * @return array
     */
    protected function quarters()
    {
        return [
            1 => 'Q1',
            2 => 'Q2',
            3 => 'Q3',
            4 => 'Q4',
        ];
    }

    /**
     * The available years from which we can choose.
     *
     * @return array
     */
    protected function years()
    {
        return range(today()->subYears(3)->year, today()->year);
    }

    /**
     * The selected type value.
     *
     * @return string
     */
    protected function selectedType()
    {
        return request('type', config('core.dashboard.comparison', 'range'));
    }

    /**
     * The selected current date.
     *
     * @param string $type
     *
     * @return Carbon
     */
    protected function selectedCurrentStart($type = 'range')
    {
        if ('range' == $type) {
            return Carbon::parse(request('current-start', today()->subDays(7)->format('Y-m-d')));
        }

        return Carbon::parse(request('current-start', today()));
    }

    /**
     * The selected previous date.
     *
     * @param string $type
     *
     * @return \Carbon\Carbon
     */
    protected function selectedPreviousStart($type = 'range')
    {
        $date = request('previous-start', null);

        if ($date) {
            return Carbon::parse($date);
        }

        $date = today();

        if ($type === 'year') {
            return $date->subYears(1)->startOfYear();
        }

        if ($type === 'quarter') {
            return $date->subQuarters(1)->startOfQuarter();
        }

        if ($type === 'range') {
            return $date->subQuarters(1)->startOfMonth();
        }

        return $date->subMonth()->startOfMonth();
    }
}
