<?php

namespace Ignite\Core\Models\Report;

use Ignite\Core\Entities\User;
use Ignite\Core\Http\Forms\ProfileForm;
use Ignite\Core\Models\Grid\ParticipantFormatter;
use Ignite\Core\Models\Grid\QueryTable;
use Ignite\Core\Traits\ReportFormConfiguration;
use Illuminate\Database\DatabaseManager;
use Yajra\DataTables\DataTables;

class EnrollmentReport extends QueryTable
{
    use ReportFormConfiguration {
        getColumns as getFormColumns;
    }

    /**
     * The columns we should always display in the report.
     * @var array
     */
    protected $alwaysDisplay = ['created_at'];

    /**
     * The form to configure report columns when using ReportFormConfiguration trait.
     * @var string
     */
    protected $form = ProfileForm::class;

    /**
     * @var ParticipantFormatter
     */
    protected $formatter;

    /**
     * EnrollmentReport constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param ParticipantFormatter $formatter
     * @param array $params
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        ParticipantFormatter $formatter,
        array $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * @return array
     */
    protected function getColumns()
    {
        $columns = $this->getFormColumns();
        unset($columns['ssn']);

        return $columns;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'user_id' => [
                'title' => 'User ID',
                'name' => 'participant.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'created_at' => [
                'title' => 'Enrollment Date',
                'name' => 'participant.created_at',
                'visible' => true,
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = collect($this->getColumns())
            ->keyBy('name')
            ->keys()
            ->toArray();

        return $this->getConnection()->table('core_participant as participant')
            ->select($columns)
            ->distinct()
            ->leftJoin('core_user as user', 'user.user_id', '=', 'participant.user_id')
            ->where('user.internal', User::TYPE_PARTICIPANT)
            ->where('user.status', User::STAT_ACTIVE);
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'agree_confirm_1' => [$this->formatter, 'agreeConfirm1'],
            'agree_confirm_2' => [$this->formatter, 'agreeConfirm2'],
            'archived' => [$this->formatter, 'archived'],
            'internal' => [$this->formatter, 'internal'],
            'status' => [$this->formatter, 'status'],
        ];
    }
}
