<?php

namespace Ignite\Core\Tests\Unit\Repositories;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Ignite\Core\Entities\User;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Tests\TestCase;
use Ignite\Core\Entities\Transaction;
use Ignite\Core\Repositories\TransactionRepository;

class TransactionRepositoriesTest extends TestCase
{
    use RefreshDatabase;

    /** @var TransactionRepository */
    private $transactionRepository;

    public function setUp() : void
    {
        parent::setUp();

        $this->transactionRepository = new TransactionRepository();
    }

    /**
     * @test
     * @group Transaction
     * @group Repository
     */
    public function it_returns_the_expected_system_transaction_types()
    {
        $types = $this->transactionRepository->getAllowedTypes();

        $this->assertInternalType('array', $types);

        $this->assertContains('CANCELLED', $types);
        $this->assertContains('EARNED', $types);
        $this->assertContains('REDEEMED', $types);
        $this->assertContains('MANUAL-RECEIVE', $types);
        $this->assertContains('MANUAL-REDEEM', $types);
    }

    /**
     * @test
     * @group Transaction
     * @group Repository
     */
    public function it_merges_configured_transaction_types()
    {
        $this->app['config']->set('transaction.types', ['FOO' => 'FOO']);
        $types = $this->transactionRepository->getAllowedTypes();

        $this->assertInternalType('array', $types);

        $this->assertContains('FOO', $types);
    }

    /**
     * @test
     * @group Transaction
     * @group Repository
     */
    public function it_provides_matching_keys_and_values_for_types()
    {
        $this->app['config']->set('transaction.types', ['BAR' => 'FOO']);
        $types = $this->transactionRepository->getAllowedTypes();

        $this->assertInternalType('array', $types);
        $this->assertArrayHasKey('FOO', $types);
        $this->assertContains('FOO', $types);
    }

    /**
     * @test
     * @group Transaction
     * @group Repository
     */
    public function it_can_get_the_balance_for_the_current_user()
    {
        $user = $this->loginAsAdmin();
        $participant = factory(Participant::class)->create([
            'user_id' => $user->getKey()
        ]);

        factory(Transaction::class)->create([
            'user_id' => $participant->getKey(),
            'value' => 1000
        ]);

        $balance = $this->transactionRepository->getBalance();
        $this->assertEquals(1000.0, $balance);
    }

    /**
     * @test
     * @group Transaction
     * @group Repository
     */
    public function it_can_get_the_balance_for_the_provided_user()
    {
        $this->loginAsAdmin();

        $user = factory(User::class)->create();
        $participant = factory(Participant::class)->create([
            'user_id' => $user->getKey()
        ]);

        factory(Transaction::class)->create([
            'user_id' => $participant->getKey(),
            'value' => 500
        ]);

        $balance = $this->transactionRepository->getBalance($participant);
        //$this->assertInternalType('float', $balance);
        $this->assertEquals(500.0, $balance);
    }
}
