<?php

namespace Ignite\Core\Models\Grid;

use Illuminate\Support\HtmlString;
use Ignite\Core\Entities\Report;

class ReportFormatter
{
    /**
     * Format the actions column.
     *
     * @param  Report $report
     * @return HtmlString
     */
    public function actions(Report $report)
    {
        if ($report->isInactive()) {
            return $this->html('<span class="text-danger">Deactivated</span>');
        }

        if (! $report->isPublished()) {
            return $this->html(sprintf(
                '<span class="text-danger">Publishing in %s %s, on %s</span>',
                $days = now()->diff($report->published_at)->days,
                str_plural('day', $days),
                $report->published_at->format('m/d/Y')
            ));
        }

        if ($report->isSensitive()) {
            return $this->html(sprintf(
                '<a class="btn btn-sm btn-default" href="%s">Send to Box</a> <span class="text-danger">Unpublishing in %s %s, on %s</span>',
                route('admin.reports.send', ['id' => $report->getKey()]),
                $days = $report->unpublished_at->diffInDays(),
                str_plural('day', $days),
                $report->unpublished_at->format('m/d/Y')
            ));
        }

        return $this->html(sprintf(
            '<a class="btn btn-sm btn-default" href="%s">Show</a>',
            route('admin.reports.show', ['id' => $report->getKey()])
        ));
    }

    /**
     * Format the status column.
     *
     * @param  Report $report
     * @return HtmlString
     */
    public function status(Report $report)
    {
        return $this->html(
            sprintf(
                '<span class="fa fa-circle %s"></span> %s',
                $report->isActive() ? 'text-green' : 'text-red',
                ucfirst($report->status)
            )
        );
    }

    /**
     * Format the created at column.
     *
     * @param  Report $report
     * @return HtmlString
     */
    public function createdAt(Report $report)
    {
        return $this->formatDate($report, 'created_at');
    }

    /**
     * Format the published at column.
     *
     * @param  Report $report
     * @return HtmlString
     */
    public function publishedAt(Report $report)
    {
        return $this->formatDate($report, 'published_at');
    }

    /**
     * Format the published at column.
     *
     * @param  Report $report
     * @return HtmlString
     */
    public function unpublishedAt(Report $report)
    {
        return $this->formatDate($report, 'unpublished_at');
    }

    /**
     * Format a date for the grid.
     *
     * @param  Report $report
     * @param  string $key
     * @return HtmlString
     */
    protected function formatDate(Report $report, $key)
    {
        return $this->html($report->getAttribute($key)->format('m/d/Y @ H:ia'));
    }

    /**
     * Format the value as a HTML string.
     *
     * @param  string $value
     * @return HtmlString
     */
    protected function html($value)
    {
        return new HtmlString($value);
    }
}
