<?php

namespace Ignite\Core\Entities;

/**
 * Core Report
 *
 * @property string $class
 */
class Report extends Base
{
    /** @const string */
    const ACTIVE = 'active';

    /** @const string */
    const INACTIVE = 'inactive';

    /**
     * The database table name.
     * @var string
     */
    protected $table = 'core_report';

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'created_at',
        'updated_at',
        'published_at',
        'unpublished_at'
    ];

    /**
     * The attributes that should be cast automatically.
     *
     * @var array
     */
    protected $casts = [
        'params' => 'array',
        'sensitive' => 'boolean',
    ];

    /**
     * Get the name of the report.
     *
     * @return string
     */
    public function getName()
    {
        return $this->getAttribute('name');
    }

    /**
     * Is the page active.
     *
     * @return bool
     */
    public function isActive()
    {
        return $this->getAttribute('status') === static::ACTIVE;
    }

    /**
     * Is the page inactive.
     *
     * @return bool
     */
    public function isInactive()
    {
        return $this->getAttribute('status') === static::INACTIVE;
    }

    /**
     * Create an instance of the report table class.
     *
     * @return \Ignite\Core\Contracts\Table
     */
    public function getReportClassInstance()
    {
        return app($this->getAttribute('class'), ['params' => $this->params]);
    }

    /**
     * Determine if the report is published.
     *
     * @return bool
     */
    public function isPublished()
    {
        return $this->published_at <= now() && $this->unpublished_at >= now();
    }

    /**
     * Determine if the report is unpublished.
     *
     * @return bool
     */
    public function isUnpublished()
    {
        return ! $this->isPublished();
    }

    /**
     * Determine if the report is sensitive.
     *
     * @return bool
     */
    public function isSensitive()
    {
        return $this->sensitive;
    }

    /**
     * The report filename.
     *
     * @param  string $prefix
     * @param  string $extension
     * @return string
     */
    public function getFilename($prefix = '', $extension = 'xlsx')
    {
        $path = empty($prefix) ? '' : rtrim($prefix, '\/\\') . DIRECTORY_SEPARATOR;
        $file = class_basename($this->class) .'_'. now()->format('YmdHis');
        $extension = ltrim($extension, '.');

        return "{$path}{$file}.{$extension}";
    }
}
