<?php

namespace Ignite\Core\Tests\Unit\Repositories;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Ignite\Core\Entities\Block;
use Ignite\Core\Tests\TestCase;

class BlockTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Set up the test.
     *
     * @return void
     */
    public function setUp() : void
    {
        parent::setUp();
    }

    /**
     * @test
     * @group Cms
     */
    public function it_can_determine_whether_it_is_active()
    {
        $block = new Block(['status' => Block::ACTIVE]);

        $this->assertTrue($block->isActive());
        $this->assertFalse($block->isInactive());
    }

    /**
     * @test
     * @group Cms
     */
    public function it_can_determine_whether_it_is_inactive()
    {
        $block = new Block(['status' => Block::INACTIVE]);

        $this->assertTrue($block->isInactive());
        $this->assertFalse($block->isActive());
    }

    /**
     * @test
     * @group Cms
     */
    public function it_can_retrieve_a_list_of_valid_statuses()
    {
        $block = new Block();
        $statuses = $block->getStatuses();

        $this->assertArrayHasKey(Block::ACTIVE, $statuses);
        $this->assertArrayHasKey(Block::INACTIVE, $statuses);
        $this->assertEquals($statuses[Block::ACTIVE], ucfirst(Block::ACTIVE));
        $this->assertEquals($statuses[Block::INACTIVE], ucfirst(Block::INACTIVE));
    }

    /**
     * @test
     * @group Cms
     */
    public function it_can_scope_a_query_for_a_block_by_code()
    {
        factory(Block::class)->create(['code' => 'test1']);
        factory(Block::class)->create(['code' => 'test2']);
        factory(Block::class)->create(['code' => 'test3']);

        $block = Block::byCode('test2')->first();

        $this->assertEquals('test2', $block->getAttribute('code'));
    }

    /**
     * @test
     * @group Cms
     */
    public function it_can_scope_a_query_to_only_active_blocks()
    {
        factory(Block::class)->create(['status' => 'active']);
        factory(Block::class)->create(['status' => 'inactive']);
        factory(Block::class)->create(['status' => 'active']);

        $blocks = Block::onlyActive()->get();

        $this->assertCount(2, $blocks);
    }

    /**
     * @test
     * @group Cms
     */
    public function it_can_scope_a_query_to_only_published_blocks()
    {
        factory(Block::class)->create(['published_at' => \Carbon\Carbon::now()->subDay(1)]);
        factory(Block::class)->create(['published_at' => \Carbon\Carbon::now()->subDay(2)]);
        factory(Block::class)->create(['published_at' => \Carbon\Carbon::now()->addDay(1)]);

        $blocks = Block::onlyPublished()->get();

        $this->assertCount(2, $blocks);
    }
}
