<?php

namespace Ignite\Core\Console;

use DomainException;
use Exception;
use Ignite\Core\Events\ImportCompleted;
use Ignite\Core\Events\ImportStarted;
use Ignite\Core\Models\Import\Importer;
use Ignite\Core\Exceptions\ImportException;
use Ignite\Core\Contracts\Repositories\ImportRepository;
use Ignite\Core\Entities\Import;
use Illuminate\Console\Command;
use Illuminate\Database\QueryException;
use Illuminate\Filesystem\FilesystemManager;
use Illuminate\Support\Facades\DB;
use Monolog\Logger;
use Symfony\Component\Console\Input\InputOption;
use League\Csv\Writer;
use Throwable;

class ImportCommand extends Command
{
    const RESULT_COLUMN = 'result';

    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:import';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import data into Ignite from a CSV file.';

    /**
     * @var ImportRepository
     */
    protected $importRepository;

    /**
     * @var FilesystemManager
     */
    private $filesystemManager;

    /**
     * Create a new command instance.
     *
     * @param ImportRepository $importRepository
     * @param FilesystemManager $filesystemManager
     */
    public function __construct(ImportRepository $importRepository, FilesystemManager $filesystemManager)
    {
        parent::__construct();

        $this->importRepository = $importRepository;
        $this->filesystemManager = $filesystemManager;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws Exception
     */
    public function handle()
    {
        $id = (int) $this->option('id');
        $user = (int) $this->option('user');
        $dryRun = $this->isDryRun();

        $flagFile = md5($id) . ".flag";
        $disk = $this->filesystemManager->disk('local');

        if (! $disk->exists($flagFile)) {
            $disk->put($flagFile, md5($flagFile));
        }

        DB::transaction(function () use ($id, $user, $dryRun) {
            $this->process($id, $user, $dryRun);
        });

        $disk->delete($flagFile);

        return 0;
    }

    /**
     * Process the import.
     *
     * @param int $id
     * @param int $user
     * @param bool $dryRun
     *
     * @return void
     * @throws Exception
     */
    protected function process($id, $user, $dryRun = false)
    {
        $import = $this->find($id);

        $imported = 0;
        $rejected = 0;
        $records = [];

        $log = $import->getLogger();

        event(new ImportStarted($import, $log));

        try {
            $importer = $import->resolveType();
            $this->info(sprintf('%s Importer', $import->getTypeLabel()));
            $this->confirm(
                sprintf(
                    'Running this process will import %s records. Are you sure you want to continue?',
                    $importer->count()
                )
            );

            if (! $dryRun) {
                $import->update(['status' => Import::STATUS_PROCESSING]);
            }

            $importer->prepare();
        } catch (Throwable $e) {
            return $this->handleException($e, $log);
        }

        foreach ($importer->process() as $line => $data) {
            $source = array_merge($data, [static::RESULT_COLUMN => Importer::RESULT_REJECTED]);

            try {
                $validator = $importer->validate($data);

                if ($validator->fails()) {
                    throw new DomainException(sprintf(
                        "Import record failed validation:\r\n%s",
                        json_encode($validator->errors()->toArray())
                    ));
                }

                $data = $importer->transform($data);

                if (! $dryRun) {
                    if ($result = $importer->save($data)) {
                        $source[static::RESULT_COLUMN] = $importer->getResultColumn() ?? Importer::RESULT_IMPORTED;
                    }
                }

                $log->info($importer->formatImportMessage($line, $data), $data);

                $imported++;
            } catch (ImportException $e) {
                $message = $importer->formatRejectMessage($line, $data, $e);
                $log->error($message, $e->toArray());
                $rejected++;
            } catch (QueryException $e) {
                $message = $importer->formatRejectMessage($line, $data, $e->getPrevious());
                $log->error(preg_replace('/Error:\sSQLSTATE\[(.*)]:\s/', '', $message), []);
                $rejected++;
            } catch (Throwable $e) {
                $message = $importer->formatRejectMessage($line, $data, $e);
                $log->error($message, $data);
                $rejected++;
            }

            $records[$line] = $source;
        }

        if ($dryRun) {
            $this->info(sprintf('%s records would be imported, %s would be rejected.', $imported, $rejected));
        } else {
            // Overwrite all the records in the original file when all records have been imported.
            try {
                $writer = Writer::createFromPath($import->getFilePath(), 'w');
                $headers = $importer->getHeaders();
                if (! in_array(static::RESULT_COLUMN, $headers)) {
                    $headers[] = static::RESULT_COLUMN;
                }
                array_unshift($records, $headers);
                $writer->insertAll($records);
            } catch (Throwable $e) {
                $this->error($e->getMessage());
            }

            $import->update([
                'run_by' => $user,
                'run_at' => now(),
                'imported' => $imported,
                'rejected' => $rejected,
                'status' => Import::STATUS_COMPLETE
            ]);

            $this->info(sprintf('%s records were imported, %s were rejected.', $imported, $rejected));
        }

        event(new ImportCompleted($import, $log));
    }

    /**
     * Determine if user asked for a dry run.
     *
     * @return bool
     */
    protected function isDryRun()
    {
        return (in_array($this->option('dry-run'), [
            true, 'true',
            1, '1',
            'yes', 'y',
            null
        ], true));
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['id', null, InputOption::VALUE_OPTIONAL, 'The ID of an existing import record.', null],
            ['dry-run', null, InputOption::VALUE_OPTIONAL, 'Should we run the import or dry-run it.', false],
            ['user', null, InputOption::VALUE_OPTIONAL, 'The user running the import.', 1]
        ];
    }

    /**
     * Find an import record by ID.
     *
     * @param  int $id
     * @return Import
     */
    protected function find($id)
    {
        return $this->importRepository->find($id);
    }

    /**
     * Handle an exception depending on the command verbosity.
     *
     * @param Throwable $e
     * @param Logger $log
     */
    protected function handleException(Throwable $e, Logger $log)
    {
        $error = sprintf('An error occurred: %s in %s on line %s', $e->getMessage(), $e->getFile(), $e->getLine());
        $log->error($error);
        $this->error($error);
    }
}
