<?php

namespace Ignite\Core\Http\Controllers\Admin;

use Ignite\Core\Models\Grid\ImportTable;
use Illuminate\Http\Request;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Http\Requests\CreateImportRequest;
use Ignite\Core\Models\Import\File;
use Ignite\Core\Contracts\Repositories\ImportRepository;
use Illuminate\Support\Facades\Storage;

class ImportController extends Controller
{
    /**
     * @var ImportRepository
     */
    protected $importRepository;

    /**
     * ImportController constructor.
     *
     * @param ImportRepository $importRepository
     */
    public function __construct(ImportRepository $importRepository)
    {
        $this->importRepository = $importRepository;
    }

    /**
     * Display a listing of the resource.
     *
     * @param ImportTable $table
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function index(ImportTable $table, Request $request)
    {
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Core::admin.import.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function create()
    {
        return redirect('admin.import.index');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  CreateImportRequest $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(CreateImportRequest $request)
    {
        try {
            $importFile = File::fromRequest(
                $request,
                $this->importRepository->getAllowedTypes()
            );

            $data = array_merge([
                'created_by' => auth()->user()->getKey()
            ], $importFile->toArray());

            $import = $this->importRepository->create($data);

            $this->flash('success', 'Your import data has been uploaded. Please check it first before importing.');

            return redirect()->route('admin.import.show', $import->getKey());
        } catch (\Exception $e) {
            $this->flash('error', $e->getMessage());

            return redirect()->back();
        }
    }

    /**
     * Show the specified resource.
     *
     * @param  int     $id
     * @param  Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id, Request $request)
    {
        $import = $this->importRepository->find($id);
        $dryRun = $request->query('dryRun', false);

        return view('Core::admin.import.show', compact('dryRun', 'import'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function destroy($id, Request $request)
    {
        try {
            $this->importRepository->delete($id);
            $status = 'success';
            $message = 'Import deleted successfully.';
        } catch (\Exception $e) {
            $status = 'error';
            $message = 'Unable to delete the import. Error: ' . $e->getMessage();
        }

        if ($request->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message
            ], 200);
        }

        $this->flash($status, $message);

        return redirect()->route('admin.import.index');
    }
}
