<?php

namespace Ignite\Core\Models\Report;

use Ignite\Core\Entities\User;
use Ignite\Core\Http\Forms\ProfileForm;
use Ignite\Core\Models\Grid\ParticipantFormatter;
use Ignite\Core\Models\Grid\QueryTable;
use Ignite\Core\Traits\ReportFormConfiguration;
use Illuminate\Database\DatabaseManager;
use Yajra\DataTables\DataTables;

class EnrollmentReport extends QueryTable
{
    use ReportFormConfiguration;

    /**
     * The form to configure report columns when using ReportFormConfiguration trait.
     * @var string
     */
    protected $form = ProfileForm::class;

    /**
     * @var ParticipantFormatter
     */
    protected $formatter;

    /**
     * EnrollmentReport constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param ParticipantFormatter $formatter
     * @param array $params
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        ParticipantFormatter $formatter,
        array $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'user_id' => [
                'title' => 'User ID',
                'name' => 'participant.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'created_at' => [
                'title' => 'Enrollment Date',
                'name' => 'participant.created_at',
                'visible' => true,
                'orderable' => true,
                'exportable' => true,
            ],
            'first' => [
                'title' => 'First',
                'name' => 'participant.first',
                'orderable' => true,
                'exportable' => true,
            ],
            'last' => [
                'title' => 'Last',
                'name' => 'participant.last',
                'orderable' => true,
                'exportable' => true,
            ],
            'type' => [
                'title' => 'Type',
                'name' => 'participant.type',
                'orderable' => true,
                'exportable' => true,
            ],
            'email' => [
                'title' => 'E-mail',
                'name' => 'participant.email',
                'orderable' => true,
                'exportable' => true,
            ],
            'company' => [
                'title' => 'Company',
                'name' => 'participant.company',
                'orderable' => true,
                'exportable' => true,
            ],
            'address_1' => [
                'title' => 'Address 1',
                'name' => 'participant.address_1',
                'orderable' => true,
                'exportable' => true,
            ],
            'address_2' => [
                'title' => 'Address 2',
                'name' => 'participant.address_2',
                'orderable' => true,
                'exportable' => true,
            ],
            'city' => [
                'title' => 'City',
                'name' => 'participant.city',
                'orderable' => true,
                'exportable' => true,
            ],
            'state' => [
                'title' => 'State',
                'name' => 'participant.state',
                'orderable' => true,
                'exportable' => true,
            ],
            'postal' => [
                'title' => 'Postal',
                'name' => 'participant.postal',
                'orderable' => true,
                'exportable' => true,
            ],
            'country' => [
                'title' => 'Country',
                'name' => 'participant.country',
                'orderable' => true,
                'exportable' => true,
            ],
            'phone1' => [
                'title' => 'Phone',
                'name' => 'participant.phone1',
                'orderable' => true,
                'exportable' => true,
            ],
            // Invisible columns
            'employee_id' => [
                'title' => 'Employee ID',
                'name' => 'participant.employee_id',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'title' => [
                'title' => 'Title',
                'name' => 'participant.title',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'phone2' => [
                'title' => 'Phone 2',
                'name' => 'participant.phone2',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'phone3' => [
                'title' => 'Phone 3',
                'name' => 'participant.phone3',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'department_1' => [
                'title' => 'Department 1',
                'name' => 'participant.department_1',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'department_2' => [
                'title' => 'Department 2',
                'name' => 'participant.department_2',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'alternate_email' => [
                'title' => 'Alternate Email',
                'name' => 'participant.alternate_email',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'region' => [
                'title' => 'Region',
                'name' => 'participant.region',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'dob' => [
                'title' => 'Date of Birth',
                'name' => 'participant.dob',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ]
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = collect($this->getColumns())
            ->keyBy('name')
            ->keys()
            ->toArray();

        return $this->getConnection()->table('core_participant as participant')
            ->select($columns)
            ->distinct()
            ->leftJoin('core_user as user', 'user.user_id', '=', 'participant.user_id')
            ->where('user.internal', User::TYPE_PARTICIPANT)
            ->where('user.status', User::STAT_ACTIVE);
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'agree_confirm_1' => [$this->formatter, 'agreeConfirm1'],
            'agree_confirm_2' => [$this->formatter, 'agreeConfirm2'],
            'archived' => [$this->formatter, 'archived'],
            'internal' => [$this->formatter, 'internal'],
            'status' => [$this->formatter, 'status'],
        ];
    }
}
