<?php

namespace Ignite\Core\Tests\Feature\Http\Controllers\Admin;

use Ignite\Core\Entities\Participant;
use Ignite\Core\Tests\Helpers\TransactionHelper;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Ignite\Core\Entities\User;
use Ignite\Core\Tests\TestCase;

class RunControllerTest extends TestCase
{
    use RefreshDatabase;
    use TransactionHelper;

    /**
     * Set up the test.
     *
     * @return void
     */
    public function setUp() : void
    {
        parent::setUp();

        $this->loadCommands([\Ignite\Core\Console\ImportCommand::class]);
        $this->loadRoutes(['Core']);
    }

    /**
     * @test
     * @group Import
     * @group Feature
     * @group ImporterDryRun
     */
    public function it_will_not_import_on_a_dry_run()
    {
        $this->withoutMiddleware();

        /** @var User $user */
        $user = $this->loginAsAdmin();

        $import = $this->setupImport("type,amount,email,description\r\nEARNED,10,it@brightspot.email,Foo#1");

        // Start the dry-run
        $response = $this->withoutMockingConsoleOutput()
            ->withoutExceptionHandling()
            ->post(route('admin.import.run', [
                'id' => $import->id,
                'dry_run' => 1
            ]))
            ->assertRedirect(route('admin.import.show', [
                'id' => $import->id,
                'dryRun' => 1,
                'onlyErrors' => 1,
                'queued' => config('core.import.queue', false)
            ]));

        $this->assertDatabaseMissing('core_transaction', [
            'user_id' => $user->getKey(),
            'value' => 10,
            'type' => 'EARNED',
            'description' => 'Foo#1',
        ]);
    }

    /**
     * @test
     * @group Import
     * @group Feature
     * @group ImporterRun
     */
    public function it_will_import_valid_data()
    {
        $this->withoutMiddleware();

        $user = $this->loginAsAdmin();

        $participant = factory(Participant::class)->create([
            'user_id' => $user->getKey(),
            'email' => 'it@brightspot.email'
        ]);

        $import = $this->setupImport("type,amount,email,description\r\nEARNED,10,it@brightspot.email,Foo#1");

        // Start the import
        $response = $this->withoutMockingConsoleOutput()
            ->withoutExceptionHandling()
            ->post(route('admin.import.run', [
                'id' => $import->id
            ]))
            ->assertRedirect(route('admin.import.show', [
                'id' => $import->id,
                'dryRun' => 0,
                'onlyErrors' => 1,
                'queued' => config('core.import.queue', false)
            ]));

        $this->assertDatabaseHas('core_transaction', [
            'user_id' => $participant->getKey(),
            'value' => 10,
            'type' => 'EARNED',
            'description' => 'Foo#1',
        ]);
    }

    /**
     * @test
     * @group Import
     * @group Feature
     * @group ImporterRun
     */
    public function it_will_import_data_for_unenrolled_participants_to_the_waiting_transactions_table()
    {
        $this->withoutMiddleware();

        $user = $this->loginAsAdmin();

        $participant = factory(Participant::class)->create([
            'user_id' => $user->getKey(),
            'email' => 'it@brightspot.email'
        ]);

        $import = $this->setupImport("type,amount,email,description\r\nEARNED,10,foo@brightspot.email,Foo#1");

        // Start the import
        $response = $this->withoutMockingConsoleOutput()
            ->withoutExceptionHandling()
            ->post(route('admin.import.run', [
                'id' => $import->id
            ]))
            ->assertRedirect(route('admin.import.show', [
                'id' => $import->id,
                'dryRun' => 0,
                'onlyErrors' => 1,
                'queued' => config('core.import.queue', false)
            ]));

        $this->assertDatabaseHas('core_transaction_resource', [
            'identifier' => 'foo@brightspot.email',
            'value' => 10,
            'type' => 'EARNED',
            'description' => 'Foo#1',
        ]);
    }
}
