<?php

namespace Ignite\Core\Tests\Feature\Http\Controllers\Admin;

use Carbon\Carbon;
use Ignite\Core\Entities\Participant;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Storage;
use Ignite\Core\Entities\User;
use Ignite\Core\Entities\Import;
use Ignite\Core\Tests\TestCase;
use Ignite\Core\Entities\Transaction;
use Ignite\Core\Services\Importers\Transactions;

class RunControllerTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Set up the test.
     *
     * @return void
     */
    public function setUp() : void
    {
        parent::setUp();

        $this->loadCommands([\Ignite\Core\Console\Import::class]);
        $this->loadRoutes(['Core']);
    }

    private function createdTimestampedFile($filename, $extension, Carbon $date, $contents)
    {
        $file = $filename . $date->format('YmdHis') . '.' . $extension;
        Storage::disk('local')->put($file, $contents);
        return $file;
    }

    private function setupImport($contents, $label = 'Transaction')
    {
        $service = Transactions::class;
        $this->app['config']->set('core.import.types.transaction', [
            'label' => 'Transaction',
            'class' => $service
        ]);
        Storage::fake('local');
        $knownDate = Carbon::create(2018, 3, 13, 12);
        Carbon::setTestNow($knownDate);
        $file = $this->createdTimestampedFile('imports/transaction-', 'csv', $knownDate, $contents);
        $log = $this->createdTimestampedFile('imports/transaction-', 'log', $knownDate, "");

        return Import::create([
            'type' => $service,
            'label' => $label,
            'file' => $file,
            'log' => $log,
        ]);
    }

    /**
     * @test
     * @group Import
     * @group Feature
     * @group ImporterDryRun
     */
    public function it_will_not_import_on_a_dry_run()
    {
        $this->withoutMiddleware();
        /** @var User $user */
        $user = $this->loginAsAdmin();

        $import = $this->setupImport("type,amount,user\r\nEARNED,10,it@brightspot.email");

        // Start the dry-run
        $this->post(route('admin.import.run', [
            'id' => $import->id,
            'dry_run' => 1
        ]))->assertRedirect(route('admin.import.show', [
            'id' => $import->id,
            'dryRun' => 1,
            'onlyErrors' => 1
        ]));

        $this->assertDatabaseMissing('core_transaction', [
            'user_id' => $user->getKey(),
            'value' => 10,
            'type' => 'EARNED'
        ]);
    }

    /**
     * @test
     * @group Import
     * @group Feature
     * @group ImporterRun
     */
    public function it_will_import_valid_data()
    {
        $this->markTestSkipped('Unable to debug.');
        $this->withoutMiddleware();
        /** @var User $user */
        $user = $this->loginAsAdmin();
        $participant = factory(Participant::class)->create([
            'user_id' => $user->getKey(),
            'email' => 'it@brightspot.email'
        ]);
        $import = $this->setupImport("type,amount,user\r\nEARNED,10,it@brightspot.email");

        //$this->followingRedirects();
        // Start the import
        $response = $this->post(route('admin.import.run', [
            'id' => $import->id
        ]))->assertRedirect(route('admin.import.show', [
            'id' => $import->id,
            'dryRun' => 0,
            'onlyErrors' => 1
        ]));

        var_dump($import->fresh());

        $this->assertDatabaseHas('core_transaction', [
            'user_id' => $participant->getKey(),
            'value' => 10,
            'type' => 'EARNED'
        ]);
    }
}
