<?php

namespace Ignite\Core\Entities;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;
use Ignite\Core\Traits\AttributeToggle;

class Block extends Model
{
    use AttributeToggle;

    /** @const string */
    const ACTIVE = 'active';

    /** @const string */
    const INACTIVE = 'inactive';

    /** @const string */
    const CODE_SEPARATOR = '_';

    /**
     * The database table name.
     * @var string
     */
    protected $table = 'core_block';

    /**
     * The fields which are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name', 'code', 'locale', 'content', 'status',
        'created_at', 'updated_at', 'published_at'
    ];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'created_at',
        'updated_at',
        'published_at'
    ];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot()
    {
        parent::boot();

        static::saving(function ($block) {
            // When saving, make sure the code slug is in the correct format
            if (isset($block->code) && !empty($block->code)) {
                $block->code = Str::slug($block->code, static::CODE_SEPARATOR);
            }

            // Purify
            $purifier = app(\Ignite\Core\Models\Purifier::class);
            $block->attributes['content'] = $purifier->clean($block->attributes['content']);
        });
    }

    /**
     * Get the statuses.
     *
     * @return array
     */
    public function getStatuses()
    {
        $statuses = [
            static::INACTIVE => ucfirst(static::INACTIVE),
            static::ACTIVE => ucfirst(static::ACTIVE),
        ];

        return $statuses;
    }

    /**
     * Is the page active.
     *
     * @return bool
     */
    public function isActive()
    {
        return $this->status === static::ACTIVE;
    }

    /**
     * Is the page inactive.
     *
     * @return bool
     */
    public function isInactive()
    {
        return $this->status === static::INACTIVE;
    }

    /**
     * Get the content as a HtmlString.
     *
     * @return HtmlString
     */
    public function getContentHtml()
    {
        return new HtmlString($this->content ?? '');
    }

    // ----- Scopes

    /**
     * Query for a URL code.
     *
     * @param  Builder $query
     * @param  string $code
     * @return Builder
     */
    public function scopeByCode(Builder $query, $code)
    {
        $query->where('code', $code);

        return $query;
    }

    /**
     * Query for a URL code.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeOnlyActive(Builder $query)
    {
        $query->where('status', static::ACTIVE);

        return $query;
    }

    /**
     * Query for a URL code.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeOnlyPublished(Builder $query)
    {
        $query->where('published_at', '<=', Carbon::now());

        return $query;
    }
}
