<?php

namespace Ignite\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Ignite\Core\Contracts\Repositories\ParticipantRepository;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Models\Grid\TrashedTable;

class TrashedController extends Controller
{
    /**
     * @var ParticipantRepository
     */
    private $participantRepository;

    /**
     * TrashedController constructor.
     *
     * @param ParticipantRepository $participantRepository
     */
    public function __construct(ParticipantRepository $participantRepository)
    {
        $this->participantRepository = $participantRepository;
    }

    /**
     * List the soft deleted participant records.
     *
     * @param  TrashedTable $table
     * @param  Request $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\View\View
     */
    public function index(TrashedTable $table, Request $request)
    {
        $status = $request->query('status', 'active');
        $statuses = $this->participantRepository->getStatuses();
        $statusLookup = array_flip($statuses);
        $current = 'trashed';

        if (! array_key_exists($status, $statusLookup)) {
            $status = 'active';
        }

        $table->with('status', $statusLookup[$status]);

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Core::admin.participants.trashed.index', compact('statuses', 'current'));
    }

    /**
     * Restore multiple soft deleted participant records.
     *
     * @param  Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        try {
            $ids = explode(',', $request->get('ids', ''));
            // TODO: Refactor and move to the ParticipantRepository
            Participant::query()
                ->onlyTrashed()
                ->whereIn('user_id', $ids)
                ->get()
                ->each(function ($participant) {
                    $participant->restore();
                });
            $message = sprintf('%s trashed %s successfully restored.',
                count($ids), str_plural('participant', count($ids))
            );
            $this->flash('success', $message);
        } catch (\Exception $e) {
            $this->flash('error', $e->getMessage());
        }

        return redirect()->route('admin.participant.trashed');
    }

    /**
     * Destroy multiple participant records.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Request $request)
    {
        $ids = explode(',', $request->get('ids', ''));
        $participants = $this->participantRepository->query()
            ->onlyTrashed()
            ->whereIn('user_id', $ids)
            ->get();

        $count = count($ids);

        try {
            foreach ($participants as $participant) {
                if (! $this->participantRepository->delete($participant->getKey())) {
                    --$count;
                }
            }

            $this->flash('success', sprintf('%s trashed %s successfully deleted.',
                $count, str_plural('participant', $count)
            ));
        } catch (\Exception $e) {
            $this->flash('error', $e->getMessage());
        }

        return redirect()->route('admin.participant.trashed');
    }
}
