<?php

namespace Ignite\Core\Http\Controllers\Admin;

use Ignite\Core\Auth\Impersonation;
use Illuminate\Http\Request;
use Ignite\Core\Entities\Note;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Models\Grid\ParticipantNotesTable;
use Ignite\Core\Contracts\Repositories\ParticipantRepository;

class ParticipantNotesController extends Controller
{
    /**
     * @var ParticipantRepository
     */
    private $participantRepository;

    /**
     * @var Impersonation
     */
    private $impersonation;

    /**
     * ParticipantOrdersController constructor.
     *
     * @param ParticipantRepository $participantRepository
     * @param Impersonation $impersonation
     */
    public function __construct(ParticipantRepository $participantRepository, Impersonation $impersonation)
    {
        $this->participantRepository = $participantRepository;
        $this->impersonation = $impersonation;
    }

    /**
     * Show the Notes that belong to the provided participant.
     *
     * @param  bool|int              $participantId
     * @param  ParticipantNotesTable $table
     * @param  Request               $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function index($participantId, ParticipantNotesTable $table, Request $request)
    {
        $table->with('user_id', $participantId);
        $participant = $this->participantRepository->find($participantId)->present();

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Core::admin.participants.notes', compact(
            'statuses',
            'current',
            'participant',
            'participantId'
        ));
    }

    /**
     * Add a Participant Note.
     *
     * @param  int $participant
     * @param  Request $request
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store($participant, Request $request)
    {
        $data = $this->validate($request, [
            'message' => 'required|string|min:10',
        ]);

        try {
            Note::create([
                'table_name' => 'participant',
                'table_primary_id' => $participant,
                'message' => $data['message'],
                'by_user_id' => auth()->user()->getKey(),
                'login_user_id' => $this->impersonation->isImpersonating()
                    ? $this->impersonation->getImpersonatorId()
                    : auth()->user()->getKey()
            ]);
            $this->flash('success', 'Note created successfully.');
            return redirect()->route('admin.participant.notes', $participant);
        } catch (\Exception $e) {
            $error = "Unable to Create Note for User Id #$participant: {$e->getMessage()}";
            $this->flash('error', $error);
            return redirect()->route('admin.participant.notes', $participant);
        }
    }
}
