<?php

namespace Ignite\Core\Models\Form;

use Illuminate\Support\Carbon;
use Kris\LaravelFormBuilder\Fields\FormField;

class Backend
{
    /** @var string */
    protected $value;

    /**
     * Backend.
     *
     * @param string $value
     */
    public function __construct($value)
    {
        $this->value = is_string($value) ? trim($value) : $value;
    }

    /**
     * Apply the type negotiation.
     *
     * @param  FormField $field
     * @return mixed
     */
    public function apply(FormField $field)
    {
        return call_user_func_array([$this, $field->getOption('backend_type')], compact('field'));
    }

    /**
     * Handle a string type.
     *
     * @param  FormField $field
     * @return string
     */
    protected function string(FormField $field)
    {
        // Possibly transliterate, swap diacritics, if necessary.

        // Check if the value is an array, if so, implode it as comma-separated
        if (is_array($this->value)) {
            $this->value = implode(',', $this->value);
        }

        return $this->value;
    }

    /**
     * Handle a password type.
     *
     * @param  FormField $field
     * @return string
     */
    protected function password(FormField $field)
    {
        return $this->value;
    }

    /**
     * Handle a textarea type.
     *
     * @param  FormField $field
     * @return string
     */
    protected function textarea(FormField $field)
    {
        // Possibly transliterate, swap diacritics, if necessary.

        return $this->value;
    }

    /**
     * Handle a blob type.
     *
     * @param  FormField $field
     * @return string
     */
    protected function blob(FormField $field)
    {
        return $this->value;
    }

    /**
     * Handle a integer type.
     *
     * @param  FormField $field
     * @return int
     */
    protected function int(FormField $field)
    {
        return (int) $this->removeNonNumericDisallowedCharacters();
    }

    /**
     * Handle a double type.
     *
     * @param  FormField $field
     * @return float
     */
    protected function double(FormField $field)
    {
        return (float) $this->removeNonNumericDisallowedCharacters();
    }

    /**
     * Handle a date type.
     *
     * @param  FormField $field
     * @return string
     */
    protected function date(FormField $field)
    {
        try {
            return Carbon::parse($this->value)->format('Y-m-d');
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * Handle a datetime type.
     *
     * @param  FormField $field
     * @return string
     */
    protected function datetime(FormField $field)
    {
        try {
            return Carbon::parse($this->value)->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * Handle a timestamp type.
     *
     * @param  FormField $field
     * @return int
     */
    protected function timestamp(FormField $field)
    {
        return Carbon::createFromTimestamp($this->value)->getTimestamp();
    }

    /**
     * Remove all non numeric characters except for a period.
     *
     * @return string
     */
    protected function removeNonNumericDisallowedCharacters()
    {
        return preg_replace('/[^0-9.]/', '', $this->value);
    }
}
