<?php

namespace Ignite\Core\Models\Menu;

use Ignite\Core\Contracts\CanDisplayInMenu;
use Ignite\Core\Events\Menus\AdminMenuPreCompile;
use Spatie\Menu\Laravel\Facades\Menu;
use Spatie\Menu\Laravel\Html;
use Spatie\Menu\Laravel\Link;
use Spatie\Menu\Laravel\Menu as SpatieMenu;

class AdminMenu extends Manager
{
    /** @var SpatieMenu */
    private $menu;

    /**
     * Add a link to the menu.
     *
     * @param CanDisplayInMenu $item
     * @param SpatieMenu $menu
     */
    public function addLink(CanDisplayInMenu $item, $menu)
    {
        $link = $menu->link($item->getUrl(), $this->buildLabel($item));
        $link->setActive(function ($link) use ($item) {
            if ($link->url() === url()->current()) {
                return true;
            }

            if (str_contains(url()->current(), $link->url()) && url($link->url()) !== url('admin')) {
                return true;
            }

            return false;
        });
    }

    /**
     * Build the label for the link,
     *
     * @param  CanDisplayInMenu $item
     * @return string
     */
    protected function buildLabel(CanDisplayInMenu $item)
    {
        return Html::raw(sprintf(
            '<i class="fa fa-fw fa-%s"></i><span>%s</span>',
            $item->getIcon() ?? 'circle-o',
            $item->getLabel()
        ))->html();
    }

    /**
     * Build a submenu link.
     *
     * @param  CanDisplayInMenu $item
     * @return Link
     */
    public function addSubmenuLink(CanDisplayInMenu $item)
    {
        $child = sprintf(
            '<i class="fa fa-fw fa-%s"></i><span>%s</span> <span class="pull-right-container"><i class="fa fa-angle-left pull-right"></i></span>',
            $item->getIcon(), $item->getLabel()
        );

        return Link::to('#', $child);
    }

    /**
     * Build the menu.
     *
     * @param  \Illuminate\Support\Collection|array $links
     * @return SpatieMenu
     */
    public function buildMenu($links)
    {
        return Menu::build($links, function ($menu, CanDisplayInMenu $item) {
            if (! $item->hasChildren()) {
                $this->addLink($item, $menu);
            } else {
                $menu->submenu(
                    $this->addSubmenuLink($item, $menu),
                    $this->buildMenu($item->getChildren())
                         ->addClass('treeview-menu')
                         ->addParentClass('treeview')
                );
            }
        });
    }

    /**
     * Render the menu.
     *
     * @return string
     */
    public function render()
    {
        event(new AdminMenuPreCompile($this));

        $this->menu = $this->buildMenu($this->getItems())
            ->addClass('sidebar-menu tree')
            ->setAttribute('data-widget', 'tree');

        return $this->menu->render();
    }
}
