<?php

namespace Ignite\Core\Models\Menu;

use Ignite\Core\Contracts\CanDisplayInMenu;
use Ignite\Core\Entities\Page;

class PageItem implements CanDisplayInMenu
{
    /** @var Page */
    private $page;

    /**
     * PageItem constructor.
     *
     * @param Page $page
     */
    public function __construct(Page $page)
    {
        $this->page = $page;
    }

    /**
     * Set the url of the view page.
     *
     * @param  string $url
     *
     * @return $this
     */
    public function setUrl($url)
    {
        $this->page->code = $url;

        return $this;
    }

    /**
     * The url of the view page.
     *
     * @return string
     */
    public function getUrl()
    {
        return url('/' . $this->page->code);
    }

    /**
     * Set the icon to show along with the label.
     *
     * @param  string $icon
     *
     * @return $this
     */
    public function setIcon($icon)
    {
        $this->page->setAttribute('icon', $icon);

        return $this;
    }

    /**
     * The icon to show along with the label.
     *
     * @return string
     */
    public function getIcon()
    {
        return $this->page->getAttribute('icon') ?? '';
    }

    /**
     * Set the page name label.
     *
     * @param  string $label
     *
     * @return $this
     */
    public function setLabel($label)
    {
        $this->page->label = $label;

        return $this;
    }

    /**
     * The page name label.
     *
     * @return string
     */
    public function getLabel()
    {
        return $this->page->label;
    }

    /**
     * Set the position of the page in the menu.
     *
     * @param  int $position
     *
     * @return $this
     */
    public function setPosition($position)
    {
        $this->page->position = $position;

        return $this->page->position;
    }

    /**
     * The position of the page in the menu.
     *
     * @return int
     */
    public function getPosition()
    {
        return $this->page->position;
    }

    /**
     * Set whether the item be viewed in the menu.
     *
     * @param  bool $flag
     *
     * @return $this
     */
    public function setCanBeViewed($flag)
    {
        $this->page->setAttribute('display_in_menu', $flag);

        return $this;
    }

    /**
     * Can the item be viewed in the menu.
     *
     * @return bool
     */
    public function canBeViewed()
    {
        return $this->page->canBeViewed();
    }

    /**
     * Determine if the item has child items.
     *
     * @return bool
     */
    public function hasChildren()
    {
        return false;
    }

    /**
     * Get the child items collection.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getChildren()
    {
        return collect([]);
    }

    /**
     * Get the value used to key the item in a collection.
     *
     * @return string
     */
    public function getMenuKey()
    {
        return '/'. $this->page->code;
    }
}