<?php

namespace Ignite\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Ignite\Core\Facades\Format;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Http\Requests\CreateTransactionRequest;
use Ignite\Core\Models\Grid\ParticipantTransactionTable;
use Ignite\Core\Repositories\ParticipantRepository;
use Ignite\Core\Repositories\TransactionRepository;

class ParticipantTransactionsController extends Controller
{
    /**
     * @var TransactionRepository
     */
    protected $transactionRepository;

    /**
     * @var ParticipantRepository
     */
    private $participantRepository;

    /**
     * ParticipantTransactionsController constructor.
     *
     * @param TransactionRepository $transactionRepository
     * @param ParticipantRepository $participantRepository
     */
    public function __construct(
        TransactionRepository $transactionRepository,
        ParticipantRepository $participantRepository
    ) {
        $this->transactionRepository = $transactionRepository;
        $this->participantRepository = $participantRepository;
    }

    /**
     * Show the Transactions that belong to the provided participant.
     *
     * @param  bool|int $userId
     * @param  ParticipantTransactionTable $table
     * @param  Request $request
     *
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function index($userId, ParticipantTransactionTable $table, Request $request)
    {
        $table->with('user_id', $userId);
        $participant = $this->participantRepository->find($userId)->present();
        $transactionSum = $this->transactionRepository->getBalance($userId);
        $balance = Format::amount($transactionSum);

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Core::admin.transactions.participant.index', compact(
            'participant', 'balance'
        ));
    }

    /**
     * Show the form to create a manual transaction.
     *
     * @param  int $userId
     * @param  int $relatedId
     * @return \Illuminate\View\View
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function create($userId, $relatedId = 0)
    {
        $transaction['user_id'] = $userId;
        $transaction['related_id']  = ($relatedId) ? $relatedId : '0';
        $formData = $transaction;
        $participant = $this->participantRepository->find($userId)->present();

        return view('Core::admin.transactions.participant.create', [
            'transactionTypes' => $this->transactionRepository->getAllowedTypes(),
            'formData'         => $formData,
            'transaction'      => $transaction,
            'participant'      => $participant,
            'cancelURI'        => '/admin/participant/view/' . $userId,
        ]);
    }

    /**
     * Create a manual transaction.
     *
     * @param  int $userId
     * @param  CreateTransactionRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store($userId, CreateTransactionRequest $request)
    {
        $formData = $request->except('_token', 'id');

        $formData['related_id'] = (int) (! empty($formData['related_id'])) ? $formData['related_id'] : 0;
        $formData['user_id'] = (int) (! empty($formData['user_id'])) ? $formData['user_id'] : $userId;

        try {
            $this->transactionRepository->create($formData);
            flash()->success('Adjustment transaction created.')->important();
        } catch (\Exception $e) {
            logger($e->getMessage());
            flash()->error($e->getMessage())->important();
            flash()->error('Adjustment transaction could not be created.')->important();
        }

        return redirect()->route('admin.participant.transactions', $userId);
    }
}
