<?php

namespace Ignite\Core\Models\Grid;

use Carbon\Carbon;
use Ignite\Core\Entities\Audit;

class HistoryFormatter extends Formatter
{
    /**
     * Format the full name of the user that added the note.
     *
     * @param  object $model
     * @return string
     */
    public function user($model)
    {
        return optional($model->user)->first . ' ' . optional($model->user)->last;
    }

    /**
     * Format the created at timestamp.
     *
     * @param  object $model
     * @return array
     */
    public function createdAt($model)
    {
        return [
            'display' => $this->formatDate($model, 'created_at', 'd/m/Y g:ia'),
            'timestamp' => $model->created_at
        ];
    }

    /**
     * Format the type of event.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function event($model)
    {
        $events = [
            'deleted' => 'danger',
            'updated' => 'warning',
            'created' => 'success',
        ];

        $event = sprintf(
            '<span class="label label-%s">%s</span>',
            $events[$model->event],
            ucwords(title_case($model->event))
        );

        return $this->html($event);
    }

    /**
     * Format the note message.
     *
     * @param Audit $history
     * @return string
     */
    public function message(Audit $history)
    {
        $entity = $history->getReadableEntityName();
        $identifier = $history->getReadableIdentifier();

        if ($history->event === 'deleted') {
            return $this->html(sprintf('%s %s <strong>%s</strong>', ucwords($history->event), $entity, $identifier));
        }

        if ($history->event !== 'updated') {
            return $this->html(sprintf('%s %s <strong>%s</strong>', ucwords($history->event), $entity, $identifier));
        }

        $history->unsetRelation('user');
        $modifications = $history->getModified();
        if (count($modifications) > 10) {
            return $this->html(view('Core::partials.audit', compact('history')));
        }

        $messages = [];
        foreach ($modifications as $attribute => $values) {
            $messages[] = sprintf(
                '<p>%s %s value %s from <strong>%s</strong> to <strong>%s</strong></p>',
                ucwords($history->event),
                $entity,
                $attribute,
                empty($values['old']) ? '""' : $values['old'],
                $this->getNewValues($values)
            );
        }

        return $this->html(implode('', $messages));
    }

    /**
     * Get the values that have changed.
     *
     * @param  array $values
     * @return string|static
     */
    private function getNewValues($values)
    {
        if (is_string($values['new'])) {
            return $values['new'];
        }

        if (is_array($values['new']) && isset($values['new']['date'])) {
            return Carbon::createFromFormat('Y-m-d H:i:s.u', $values['new']['date']);
        }

        return json_encode($values['new']);
    }
}
