<?php

namespace Ignite\Core\Tests\Unit\Models\Stats;

use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\User;
use Ignite\Core\Models\Dashboard\Charts\ParticipantsPendingStat;
use Ignite\Core\Models\Dashboard\Charts\TransactionTypeDonut;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Ignite\Core\Models\Dashboard\Charts\ParticipantsEnrolledStat;
use Ignite\Core\Models\Dashboard\Manager;
use Ignite\Core\Tests\TestCase;

class ManagerTest extends TestCase
{
    use RefreshDatabase;

    public function setUp() : void
    {
        parent::setUp();
    }

    /**
     * @test
     * @group Admin
     */
    public function it_allows_stat_classes_to_be_added_to_a_collection()
    {
        $manager = new Manager();

        $stat = new ParticipantsEnrolledStat([
            'color' => 'yellow',
            'icon' => 'ion ion-social-usd',
            'link' => 'foo',
        ]);

        $manager->add($stat);

        $this->assertCount(1, $manager);
        $this->assertSame('yellow', $stat->getColor());
    }

    /**
     * @test
     * @group Admin
     */
    public function it_can_calculate_data_for_all_registered_charts_at_once()
    {
        factory(User::class, 10)->create([
            "status" => User::STAT_ACTIVE,
            "internal" => User::TYPE_PARTICIPANT,
        ]);
        factory(User::class, 5)->create([
            "status" => User::STAT_PENDING,
            "internal" => User::TYPE_PARTICIPANT,
        ]);

        $manager = new Manager();
        $stat1 = new ParticipantsEnrolledStat([
            'label' => 'Participants Enrolled',
            'color' => 'blue',
            'icon'  => 'ion ion-ios-personadd-outline',
        ]);

        $stat2 = new ParticipantsPendingStat([
            'label' => 'Participants Pending',
            'color' => 'aqua',
            'icon'  => 'ion ion-ios-person-outline',
        ]);

        $manager->addMany([$stat1, $stat2])->calculate();

        $this->assertEquals(10, $manager->get('participants-enrolled-correlated-stat')->get('data'));
        $this->assertEquals(5, $manager->get('participants-pending-correlated-stat')->get('data'));
    }

    /**
     * @test
     * @group Admin
     */
    public function it_can_filter_to_only_stats()
    {
        $manager = new Manager();

        $stat = new ParticipantsEnrolledStat([
            'color' => 'yellow',
            'icon' => 'ion ion-social-usd',
            'link' => url('/admin/claim/list/issued'),
        ]);

        $manager->add($stat);

        $this->assertCount(1, $manager->stats());
    }

    /**
     * @test
     * @group Admin
     */
    public function it_can_filter_to_only_donuts()
    {
        $manager = new Manager();

        $stat = new TransactionTypeDonut([
            'colors' =>  ['#f39c12', '#00c0ef', '#00a65a', '#dd4b39'],
            'icon' => 'fa fa-gifts',
            'link' => 'foo',
        ]);

        $manager->add($stat);

        $this->assertCount(1, $manager->donuts());
    }
}
