<?php

namespace Ignite\Core\Models\Grid;

use Illuminate\Database\DatabaseManager;
use Ignite\Core\Entities\Audit;
use Yajra\DataTables\DataTables;

class ParticipantHistoryTable extends EloquentTable
{
    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [];

    /**
     * @var ParticipantFormatter
     */
    protected $formatter;

    /**
     * ParticipantTable constructor.
     *
     * @param DataTables       $datatables
     * @param DatabaseManager  $databaseManager
     * @param array            $params
     * @param HistoryFormatter $formatter
     */
    public function __construct(DataTables $datatables, DatabaseManager $databaseManager, array $params = [], HistoryFormatter $formatter)
    {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'created_at' => [
                'title' => 'Created',
                'name' => 'created_at',
                'orderable' => true,
                'exportable' => true,
                'render' => function() {
                    return ['_' => 'display', 'sort' => 'timestamp'];
                }
            ],
            'user_id' => [
                'title' => 'Changed By',
                'name' => 'user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'event' => [
                'title' => 'Event',
                'name' => 'event',
                'orderable' => true,
                'exportable' => true,
            ],
            'old_values' => [
                'title' => 'Message',
                'name' => 'old_values',
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        return $this->applyScopes(
            Audit::with('user')->where('context_type', \Ignite\Core\Entities\Participant::class)
                               ->where('context_id', $this->user_id)
        );
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'created_at' => [$this->formatter, 'createdAt'],
            'user_id' => [$this->formatter, 'user'],
            'event' => [$this->formatter, 'event'],
            'old_values' => [$this->formatter, 'message'],
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters()
    {
        $params = parent::getBuilderParameters();

        return array_merge($params, [
            'order' => [[2, 'desc']],
            'buttons' => [
                'pageLength',
                $this->getColumnVisibilityButton(),
                'csv'
            ],
        ]);
    }
}
