<?php

namespace Ignite\Core\Http\Controllers\Auth;

use Ignite\Core\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Ignite\Core\Entities\User;

class LoginController extends Controller
{
    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */
    protected $redirectTo = '/';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest', ['except' => 'logout']);
    }

    /**
     * Overrides the default get login username to be used by the controller.
     *
     * @return string
     */
    public function username()
    {
        return 'username';
    }

    /**
     * Overrride Laravel's default login form.
     *
     * @return \Illuminate\Http\Response
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Attempt to log the user into the application if the user is active.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return bool
     */
    protected function attemptLogin(Request $request)
    {
        $credentials = $this->credentials($request);

        $useEmail = config('participant.use_email_for_username');
        if ($useEmail) {
            $credentials['email'] = $credentials['username'];
            unset($credentials['username']);
        }

        return $this->guard()->attempt(
            array_merge(['status' => User::STAT_ACTIVE], $credentials),
            $request->has('remember')
        );
    }

    /**
     * Get the failed login response instance.
     *
     * @param  \Illuminate\Http\Request  $request
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function sendFailedLoginResponse(Request $request)
    {
        $user = User::where('email', $request->only($this->username()))->first();

        if ($user && $user->isInactive()) {
            throw ValidationException::withMessages(['status' => 'Your account is inactive.']);
        }

        if ($user && $user->isPending()) {
            throw ValidationException::withMessages([
                'status' => 'Your account is pending. Please set your password. '
                          . 'If you have not received an email with instructions on setting your password. '
                          . 'Please reset your password.'
            ]);
        }

        throw ValidationException::withMessages([
            $this->username() => [trans('auth.failed')],
        ]);
    }

    /**
     * This will be used instead of the AuthenticateUsers::logout() trait
     * see: use AuthenticatesUsers above
     * Log the user out of the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        $this->guard()->logout();

        $request->session()->invalidate();

        flash()->success('You have signed-out.');

        return redirect('/');
    }
}
