<?php

namespace Ignite\Claim\Models\Grid;

use Illuminate\Database\DatabaseManager;
use Ignite\Core\Models\Grid\EloquentTable;
use Ignite\Claim\Entities\OfferPromotion;
use Yajra\DataTables\DataTables;

class PromotionTable extends EloquentTable
{
    /**
     * @var ClaimFormatter
     */
    protected $formatter;

    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [
        'html'
    ];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [
        'actions', 'id', 'name', 'description', 'start_date', 'end_date', 'status'
    ];

    /**
     * PromotionTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param array $params
     * @param PromotionFormatter|null $formatter
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        array $params = [],
        PromotionFormatter $formatter = null
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'action' => [
                'name' => 'actions',
                'title' => 'Actions',
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'visible' => true,
                'width' => '130px',
                'class' => 'actions'
            ],
            'id' => [
                'title' => 'Promotion ID',
                'name' => 'claim_offer_promotion.id',
                'orderable' => true,
                'exportable' => true,
                'width' => '100px',
            ],
            'name' => [
                'title' => 'Name',
                'name' => 'claim_offer_promotion.name',
                'orderable' => true,
                'exportable' => true,
            ],
            'description' => [
                'title' => 'Description',
                'name' => 'claim_offer_promotion.description',
                'orderable' => true,
                'exportable' => true,
            ],
            'start_date' => [
                'title' => 'Start Date',
                'name' => 'claim_offer_promotion.start_date',
                'orderable' => true,
                'exportable' => true,
            ],
            'end_date' => [
                'title' => 'End Date',
                'name' => 'claim_offer_promotion.end_date',
                'orderable' => true,
                'exportable' => true,
            ],
            'status' => [
                'title' => 'Status',
                'name' => 'claim_offer_promotion.status',
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The promotions query.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        $columns = collect($this->getColumns())->keyBy('name')->except('actions')->keys()->toArray();

        $query = OfferPromotion::query()->onlyActive();

        if ((int) $this->user_id !== 0) {
            $query->where('claim_participant.user_id', $this->user_id);
        }

        return $this->applyScopes($query);
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'action' => function ($model) { return $this->formatter->actions($model); },
            'status' => function ($model) { return $this->formatter->status($model); },
            'start_date' => function ($model) { return $this->formatter->startDate($model); },
            'end_date' => function ($model) { return $this->formatter->endDate($model); },
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters(): array
    {
        $params = parent::getBuilderParameters();

        return array_merge($params, [
            'order'   => [[2, 'asc']],
            'buttons' => [
                'pageLength',
                $this->getColumnVisibilityButton(),
                'postCsv'
            ],
        ]);
    }
}
