<?php

namespace Ignite\Claim\Models\Grid;

use Illuminate\Database\DatabaseManager;
use Ignite\Claim\Entities\ClaimParticipant;
use Ignite\Core\Models\Grid\EloquentTable;
use Yajra\DataTables\DataTables;

class ProcessTable extends EloquentTable
{
    /** @var ProcessFormatter */
    protected $formatter;

    /**
     * ProcessTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param array $params
     * @param ProcessFormatter $formatter
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        array $params = [],
        ProcessFormatter $formatter = null
    ) {
        parent::__construct($datatables, $databaseManager, $params, null);

        $this->formatter = $formatter ?? new ProcessFormatter();
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns(): array
    {
        return [
            'checkbox' => [
                'defaultContent' => '',
                'title'          => '&nbsp;',
                'name'           => 'checkbox',
                'targets'        => 0,
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => true,
                'width'          => '10px',
                'order'          => 1,
            ],
            'action' => [
                'title' => 'Actions',
                'name' => 'actions',
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'width' => '90px',
                'class' => 'actions',
            ],
            'claim_id' => [
                'title' => 'Claim ID',
                'name' => 'claim_participant.claim_id',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
                'width' => '70px'
            ],
            'date_submitted' => [
                'title' => 'Submitted Date',
                'name' => 'claim_participant.date_submitted',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
                'width' => '100px'
            ],
            'status' => [
                'title' => 'Status',
                'name' => 'claim_participant.status',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
            ],
            'participant.type' => [
                'title' => 'Participant Type',
                'name' => 'participant.type',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
            ],
            'participant.first' => [
                'title' => 'First Name',
                'name' => 'participant.first',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
            ],
            'participant.last' => [
                'title' => 'Last Name',
                'name' => 'participant.last',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
            ],
            'participant.email' => [
                'title' => 'Participant Email',
                'name' => 'participant.email',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
            ],
            'claim.customer_name' => [
                'title' => 'Customer Name',
                'name' => 'claim.customer_name',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
            ],
            'claim.revenue_sold' => [
                'title' => 'Revenue Sold',
                'name' => 'claim.revenue_sold',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
                'className' => 'text-right',
                'width' => '90px'
            ],
            'value'  => [
                'title' => 'Total Amount',
                'name' => 'value',
                'searchable' => true,
                'orderable' => true,
                'exportable' => true,
                'printable' => true,
                'className' => 'text-right',
                'width' => '90px'
            ],
        ];
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'checkbox' => function ($model) { return $this->formatter->checkbox($model); },
            'action' => function ($model) { return $this->formatter->actions($model); },
            'status' => function ($model) { return $this->formatter->status($model); },
            'revenue_sold' => function ($model) { return $this->formatter->revenue($model); },
            'date_submitted' => function ($model) { return $this->formatter->submittedAt($model); },
            'date_approved' => function ($model) { return $this->formatter->approvedAt($model); },
            'value' => function ($model) { return $this->formatter->value($model); },
        ];
    }

    /**
     * The table query.
     *
     * @return \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Query\Builder|\Illuminate\Support\Collection
     */
    public function query()
    {
        $query = ClaimParticipant::with(['participant', 'claim'])
                                 ->where('claim_participant.status', 'pending');

        return $this->applyScopes($query);
    }

    /**
     * Get the parameters for building the datatable.
     *
     * @return array
     */
    protected function getBuilderParameters(): array
    {
        $params = parent::getBuilderParameters();

        return array_merge($params, [
            'buttons' => [
                'pageLength',
                //'postCsv'
            ],
            "order" => [3, 'asc'],
            'select' => [
                'info' => true,
                'style' => 'multi',
                'selector' => 'td:first-child'
            ],
            'columnDefs' => [
                ['targets' => 0, 'checkboxes' => ['selectRow' => true]]
            ],
            'stateSave' => false
        ]);
    }
}
