<?php

namespace Ignite\Claim\Models\Grid;

use Illuminate\Database\DatabaseManager;
use Ignite\Core\Models\Grid\QueryTable;
use Yajra\DataTables\DataTables;

class ClaimTable extends AbstractClaimTable
{
    /**
     * @var ClaimFormatter
     */
    protected $formatter;

    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [
        'form.claim.partial.claim_fields_external',
        'form.claim.partial.claim_fields_internal',
    ];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [
        'actions', 'claim_id', 'date_submitted', 'status', 'first', 'last',
        'participant_type', 'participant_email', 'value'
    ];

    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        array $params = [],
        ClaimFormatter $formatter = null
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns(): array
    {
        return [
            'action' => [
                'name' => 'actions',
                'title' => 'Actions',
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'visible' => true,
                'width' => '80px',
                'class' => 'actions'
            ],
            'claim_id' => [
                'title' => 'Claim ID',
                'name' => 'claim_participant.claim_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'date_submitted' => [
                'title' => 'Date Submitted',
                'name' => 'claim_participant.date_submitted',
                'orderable' => true,
                'exportable' => true,
            ],
            'status' => [
                'title' => 'Status',
                'name' => 'claim_participant.status',
                'orderable' => true,
                'exportable' => true,
            ],
            'first' => [
                'title' => 'First',
                'name' => 'participant.first',
                'orderable' => true,
                'exportable' => true,
            ],
            'last' => [
                'title' => 'Last',
                'name' => 'participant.last',
                'orderable' => true,
                'exportable' => true,
            ],
            'participant_type' => [
                'title' => 'Type',
                'name' => 'claim_participant.participant_type',
                'orderable' => true,
                'exportable' => true,
            ],
            'participant_email' => [
                'title' => 'E-mail',
                'name' => 'claim_participant.participant_email',
                'orderable' => true,
                'exportable' => true,
            ],
            'value' => [
                'title' => 'Value',
                'name' => 'claim_participant.value',
                'orderable' => true,
                'exportable' => true,
                'visible' => true,
            ],
            'user_id' => [
                'title' => 'User ID',
                'name' => 'claim_participant.user_id',
                'orderable' => true,
                'exportable' => true,
                'visible' => false,
            ],
            'id' => [
                'title' => 'Claim Participant ID',
                'name' => 'claim_participant.id',
                'orderable' => true,
                'exportable' => true,
                'visible' => false,
            ],
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        $columns = collect($this->getColumns())->keyBy('name')->except('actions')->keys()->toArray();

        $query = $this->getConnection()->table('claim_participant')
            ->select($columns)
            ->distinct()
            ->leftJoin('claim', 'claim.id', '=', 'claim_participant.claim_id')
            ->leftJoin('core_participant AS participant', 'participant.user_id', '=', 'claim_participant.user_id')
            ->leftJoin('core_user', 'core_user.user_id', '=', 'claim_participant.user_id')
            ->whereNull('claim_participant.deleted_at');

        if ($this->status !== 'all') {
            $query->where('claim_participant.status', $this->status);
        }

        if ((int) $this->user_id !== 0) {
            $query->where('claim_participant.user_id', $this->user_id);
        }

        return $this->applyScopes($query);
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'action' => function ($model) { return $this->formatter->actions($model); },
            'status' => function ($model) { return $this->formatter->status($model); },
            'date_submitted' => function ($model) { return $this->formatter->submittedAt($model); },
            'activity_date' => function ($model) { return $this->formatter->activityAt($model); },
            'date_canceled' => function ($model) { return $this->formatter->canceledAt($model); },
            'date_declined' => function ($model) { return $this->formatter->declinedAt($model); },
            'date_approved' => function ($model) { return $this->formatter->approvedAt($model); },
            'date_issued' => function ($model) { return $this->formatter->issuedAt($model); },
            'participant_email' => function ($model) { return $this->formatter->participantEmail($model); },
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters(): array
    {
        $params = parent::getBuilderParameters();

        return array_merge($params, [
            'order'   => [[2, 'asc']],
            'buttons' => [
                'pageLength',
                $this->getColumnVisibilityButton(),
                //'postCsv'
            ],
        ]);
    }
}
