<?php

namespace Ignite\Claim\Services;

use Ignite\Claim\Entities\ClaimParticipant;
use Ignite\Claim\Models\Status;
use Ignite\Claim\Entities\Offer;
use Ignite\Claim\Models\Form;
use Ignite\Claim\Repositories\ClaimRepository;
use Ignite\Core\Exceptions\ImportException;
use Ignite\Core\Contracts\Importable;
use Ignite\Core\Models\Import\Importer as BaseImporter;
use Illuminate\Support\HtmlString;
use Illuminate\Validation\Validator;
use Throwable;

class Importer extends BaseImporter implements Importable
{
    /** @var string */
    protected $formKey = 'form.claim.partial.claim_fields_external';

    /**
     * The list of "known" fields.
     *
     * @var array
     */
    protected $fields = [
        'id' => ['claim_id'],
        'user_id' => ['email', 'account', 'user', 'employee_id', 'submitter'],
        'offer_promotion_id' => ['offer', 'promotion'],
        'claim_participants' => ['splits', 'additional', 'participants'],
        'claim_items' => ['offers', 'promotions', 'lineitems']
    ];

    /**
     * @var array
     */
    protected $formSettings = [];

    /**
     * @var ClaimRepository
     */
    protected $claimRepository;

    /**
     * Importer constructor.
     *
     * @param ClaimRepository $claimRepository
     */
    public function __construct(ClaimRepository $claimRepository)
    {
        $this->claimRepository = $claimRepository;
    }


    /**
     * Return the column header in the csv file to use in order to identify the user.
     *
     * @return string
     */
    protected function getUserIdentifier()
    {
        return 'email';
    }

    /**
     * The column in the database to look up for the user.
     *
     * @return string
     */
    protected function getUserColumn()
    {
        return 'user_id';
    }

    /**
     * Display the information text as html while previewing the data.
     *
     * @return string
     */
    public function html() : string
    {
        $text = 'Claim data can be imported without additional participants and without offer line items. Be sure to check the offers and rules for the program before importing.';
        return new HtmlString(sprintf('<div class="alert alert-danger alert-dismissible bg-red-gradient"><button type="button" class="close" data-dismiss="alert" aria-hidden="true">×</button><span class="fa fa-info-circle"></span> %s</div>', $text));
    }

    /**
     * Prepare any dependencies for each iteration of the import process.
     *
     * @return void
     * @throws \League\Csv\Exception
     */
    public function prepare()
    {
        $this->cacheUsers($this->getUserColumn(), $this->getUserIdentifier());
        $this->cacheOffers();
        $this->record->deleteLog();
    }

    /**
     * Validate a single line.
     *
     * @param  array $data
     * @return Validator
     */
    public function validate(array $data) : Validator
    {
        return Form::buildValidatorInstance('claim', $data, $this->getFormSettings($this->formKey));
    }

    /**
     * Transform the data for the given line.
     *
     * @param  array $line
     * @return array
     */
    public function transform(array $line) : array
    {
        $line[$this->getUserIdentifier()] = $line[$this->getUserColumn()] ?? null;
        $line[$this->getUserColumn()] = $this->transformUser($line, $this->getUserColumn());

        // Loop through the form meta data to find date fields and transform their values
        foreach ($this->getFormSettingFor($this->formKey, 'claim') as $key => $setting) {
            $name = $setting['displayName'];
            if (array_key_exists($name, $line)) {
                $line[$key] = $line[$name];
                unset($line[$name]);
            }
            if (array_key_exists($key, $line)) {
                if ($setting['type'] === 'date') {
                    $line[$key] = $this->transformDate($line, $key);
                }
            }
        }

        if (! empty($line['claim_participants'])) {
            $participants = $this->transformParticipants($line);
            unset($line['claim_participants']);
            $line['claim_participants'] = $participants;
        } else {
            $line['claim_participants'] = [];
        }

        if (! empty($line['claim_items'])) {
            $lineitems = $this->transformLineitems($line);
            unset($line['claim_items']);
            $line['claim_items'] = $lineitems;
        } else {
            $line['claim_items'] = [];
        }

        return $line;
    }

    /**
     * Process the data for a single line.
     *
     * @param  array $data
     *
     * @return bool
     * @throws ImportException
     */
    public function save(array $data) : bool
    {
        $data['class'] = 'claim';
        $data['upload_files'] = [];

        if (isset($data['id']) && !empty(trim($data['id']))) {
            $data = $this->prepareUpdate($data);
        } else {
            $data = $this->prepareCreate($data);
        }

        $response = $this->claimRepository->put($data);
        $this->handleUpdateValue($data, $response->data);
        $this->handleUpdateStatus($data, $response->data);

        if (! $response->result) {
            logger()->error($response->errors);
            throw new ImportException('Unable to save claim data', $response->errors);
        }

        return (bool) $response->result;
    }

    /**
     * The claim participant model.
     *
     * @param  array $data
     * @return ClaimParticipant
     */
    protected function getClaimParticipant($data)
    {
        return ClaimParticipant::where('claim_id', $data['id'])
            ->where('user_id', $data['user_id'])
            ->first();
    }

    /**
     * Prepare the data for creating a new claim.
     *
     * @param  array $data
     * @return array
     */
    protected function prepareCreate(array $data) : array
    {
        $data['mode'] = 'new';
        $data['action'] = 'createClaim';

        return $data;
    }

    /**
     * Prepare the data for updating an existing claim.
     *
     * @param  array $data
     * @param  array $response
     * @return array
     */
    protected function prepareUpdate(array $data, $response = []) : array
    {
        $data['mode'] = 'edit';
        $data['action'] = 'updateClaim';
        $data['claim_id'] = $data['id'];

        $claimParticipant = $this->getClaimParticipant(empty($response) ? $data : $response);
        $data = array_merge($claimParticipant->claim->toArray(), $data);

        if ($claimParticipant) {
            $data['claim_participant'] = [
                'id' => $claimParticipant->id,
                'created_at' => $claimParticipant->created_at,
            ];

            $data['claim_items'] = $claimParticipant['lineitems']->map(function ($item) {
                return $item;
            });
        }

        unset($data['claim_participants']);

        return $data;
    }

    /**
     * Handle updating the status.
     *
     * @param  array $data
     * @param  array $response
     * @return bool
     * @throws ImportException
     */
    protected function handleUpdateStatus($data, $response)
    {
        if (! isset($data['status'])) {
            return false;
        }

        $status = strtolower($data['status']);
        $claimParticipant = $this->getClaimParticipant(empty($response) ? $data : $response);

        if (strtolower($claimParticipant->status) === $status) {
            return false;
        }

        if (! in_array($status, ClaimParticipant::$statuses)) {
            throw new ImportException("Attempted to import an unknown claim status: $status.");
        }

        /** @var Status $statusHelper */
        $statusHelper = app(Status::class);
        $statusHelper->setStatus($claimParticipant->id, [
            'status' => $status,
            'reason' => $data['reason_declined'] ?? ''
        ]);

        return true;
    }

    /**
     * Handle updating the status.
     *
     * @param  array $data
     * @param  array $response
     * @return bool
     */
    protected function handleUpdateValue($data, $response)
    {
        if (isset($data['value'])) {
            $claimParticipant = $this->getClaimParticipant(empty($response) ? $data : $response);
            $claimParticipant->forceFill([
                'value' => (int) $data['value'],
                'value_calculated' => (int) $data['value'],
                'value_adjust' => 0
            ])->save();
        }

        return true;
    }

    /**
     * Add a success message to the log file.
     *
     * @param  int   $line
     * @param  array $data
     * @return string
     */
    public function formatImportMessage(int $line, array $data) : string
    {
        return sprintf('Imported claim on line `%s`', $line);
    }

    /**
     * Add an error message to the log file.
     *
     * @param  int        $line
     * @param  array      $data
     * @param  \Exception $error
     * @return string
     */
    public function formatRejectMessage(int $line, array $data, Throwable $error) : string
    {
        $message = $this->formatRejectError($error);

        return sprintf('Rejected claim on line `%s` => %s', $line, $message);
    }

    /**
     * Format the error message.
     *
     * @param  Throwable  $error
     * @return string
     */
    protected function formatRejectError(Throwable $error) : string
    {
        $message = $error->getMessage();
        $messages = explode("\n", $message);

        // if message is two lines, with the second line being a JSON string
        if (2 === count($messages)) {
            $errLines = json_decode($messages[1], true);
            if (null !== $errLines) {
                $newErrs = [];
                foreach ($errLines as $key => $values) {
                    foreach ($values as $s) {
                        // $newErrs[] = "{$key} - {$s}";
                        $newErrs[] = $s;
                    }
                }
                $message = $messages[0] . ' ' . implode(' | ', $newErrs);
            }
        }

        return $message;
    }

    /**
     * Parse the participant data and transform it into something usable.
     *
     * @param  array $line
     * @return array
     */
    protected function transformParticipants($line)
    {
        return $this->parseMultiLine($line['claim_participants'], function ($line) {
            return array_combine(['participant_email', 'participant_type'], $line);
        });
    }

    /**
     * Parse the line items data and transform it into something usable.
     *
     * @param  array $line
     * @return array
     */
    protected function transformLineitems($line)
    {
        return $this->parseMultiLine($line['claim_items'], function ($item) {
            if (array_key_exists($item[1], $this->cache['offers'])) {
                $offerId = $this->cache['offers'][$item[1]];
            } else {
                throw new \DomainException("Line item error: Offer with name `{$item[1]}` could not be found.");
            }

            $item[] = $offerId;

            return array_combine(['offer_group', 'name', 'qty', 'offer_id'], $item);
        });
    }

    /**
     * The settings for the claim form.
     *
     * @param  string $key
     * @return array|\Ignite\Core\Entities\Form
     */
    protected function getFormSettings($key)
    {
        if (! array_key_exists($key, $this->formSettings) || empty($this->formSettings[$key])) {
            $this->formSettings[$key] = \Ignite\Claim\Entities\Form::findByKey($key);
        }

        return $this->formSettings[$key];
    }

    /**
     * The settings for the form key and internal type.
     *
     * @param  string $key
     * @param  string $type
     * @return array
     */
    protected function getFormSettingFor($key, $type)
    {
        $settings = $this->getFormSettings($key);

        return collect($settings->columns[$type])->sortBy('order')->toArray();
    }

    /**
     * Parse multiple lines of semi-colon delimited data.
     *
     * @param  string        $line
     * @param  callable|null $callback
     * @return array
     */
    protected function parseMultiLine($line, callable $callback = null)
    {
        if (false !== strpos($line, ';')) {
            $lines = preg_split("/\s?;\s?/", $line);
            return array_map(function ($item) use ($callback) {
                if (is_callable($callback)) {
                    return $callback($this->parseLine($item));
                }
                return $this->parseLine($item);
            }, $lines);
        }

        return [$callback($this->parseLine($line))];
    }

    /**
     * Parse a single line of pipe delimited data.
     *
     * @param  string $row=
     * @return array
     */
    protected function parseLine($row)
    {
        return preg_split('/\s?\|\s?/', $row);
    }

    /**
     * Cache the offers for lookup during transform.
     *
     * @return array
     */
    protected function cacheOffers()
    {
        if (! isset($this->cache['offers']) || empty($this->cache['offers'])) {
            $this->cache['offers'] = Offer::all()
                ->keyBy('name')
                ->map(function (Offer $offer) {
                    return $offer->getKey();
                })
                ->toArray();
        }
        return $this->cache['offers'];
    }
}
