<?php

namespace Ignite\Claim\Http\Controllers;

use App;
use Ignite\Claim\Entities\ClaimParticipant;
use Ignite\Flash\Facades\Flash;
use Illuminate\Support\Facades\Validator;
use View;
use Ignite\Core\Entities\User;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Claim\Entities\Setting;
use Ignite\Claim\Models\Form;
use Ignite\Claim\Entities\Claim;
use Ignite\Claim\Models\ClaimUtil;
use Ignite\Claim\Repositories\ClaimRepository;
use Ignite\Claim\Entities\OfferCampaign;
use Ignite\Claim\Entities\OfferPromotion;
use Ignite\Core\Repositories\ParticipantRepository;

class ClaimController extends Controller
{
    /** @var ClaimRepository */
    protected $claimRepository;

    /** @var ParticipantRepository */
    protected $participantRepository;

    /** @var ClaimUtil */
    protected $claimUtil;

    /** @var string */
    protected static $claimFieldsExternalKey = 'form.claim.partial.claim_fields_external';

    /** @var string */
    protected static $claimParticipantFieldsExternalKey = 'form.claim.partial.claim_fields_internal';

    /**
     * Create a new controller instance.
     *
     * @param ClaimRepository $claimRepository
     * @param ParticipantRepository $participantRepository
     */
    public function __construct(
        ClaimRepository $claimRepository,
        ParticipantRepository $participantRepository
    ) {
        $this->claimRepository = $claimRepository;
        $this->participantRepository = $participantRepository;
        // @todo Refactor mismanaged dependency -
        $this->claimUtil = new ClaimUtil($this->participantRepository);
    }

    /**
     * Show the Claim Status List.
     *
     * @param  string $status
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $status = strtolower(trim(request('status', '')));
        $claimParticipants = ClaimParticipant::where('user_id', auth()->user()->getKey());

        if (! empty($status)) {
            $claimParticipants->where('status', $status);
        }

        return view('claim.index', [
            'status' => $status,
            'claimParticipants' => $claimParticipants->get(),
        ]);
    }

    /**
     * Show the Review New Claim entry page.
     *
     * @return \Illuminate\View\View
     */
    public function review()
    {
        $errors = [];
        $formData = request()->all();

        // Save Form Data for re-display and Saving to Database
        $formData['class']              = 'claim';
        $formData['claim_items']        = request('claim_items', []);
        $formData['claim_participants'] = request('claim_participants', []);
        $formData['upload_files']       = request('upload_files', []);
        $claim = $formData['dynamicFields']['claim'] ?? [];

        foreach ($formData as $key => $value) {
            if (isset($claim[$key]) && ! empty($claim[$key])) {
                $formData[$key] = $claim[$key];
            }
        }

        $claim              = $this->claimRepository->data($formData);
        $claimItems         = $formData['claim_items'];
        $uploadFiles        = $formData['upload_files'];
        $claimParticipants  = $formData['claim_participants'];
        $promotionId        = $formData['offer_promotion_id'];

        unset($formData['dynamicFields']);

        $code = request('code', uniqid(''));
        session()->put($code, serialize($formData));

        if (! isset($formData['maxLineitems'])) {
            $formData['maxLineitems'] = setting('claim.max_lineitems');
        }

        if (! isset($formData['maxParticipants'])) {
            $formData['maxParticipants'] = setting('claim.max_participants');
        }

        $mode = $formData['mode'];

        if ($mode == 'new') {
            $returnURI = '/claim/new/' . $promotionId . '/' . $code;
        } else {
            $returnURI = '/claim/edit/' . $formData['claim_id'] . '/' . $code;
        }

        // ---------------------------------
        // Get Claim Submitter Participant Info
        // ---------------------------------
        $submitter   = $this->claimUtil->getSubmitterParticipant($errors);

        if (! $submitter) {
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        // ------------------------------
        // Get Form Fields
        // ------------------------------
        $claimFormSettings = \Ignite\Claim\Entities\Form::findByKey(self::$claimFieldsExternalKey);

        if (! $claimFormSettings) {
            $errors[] = sprintf("Missing claim form configuration. Key = '%s'\n", self::$claimFieldsExternalKey);
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        $claimParticipantFormSettings = \Ignite\Claim\Entities\Form::findByKey(self::$claimParticipantFieldsExternalKey);

        if (! $claimParticipantFormSettings) {
            $errors[] = sprintf("Missing claim participant form configuration. Key = '%s'\n", self::$claimParticipantFieldsExternalKey);
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        // ---------------------------------
        // Validate Claim data
        // ---------------------------------
        $errors = Form::validateFormData('claim', $claim, $claimFormSettings);

        if (empty($errors)) {
            // $errors = Form::validateFormData('claim_participant', $claimParticipants, $claimParticipantFormSettings);
            $errors = Form::validateFormData('claim_participant', $claimParticipants, $claimFormSettings);
        }

        // ---------------------------------
        // Validate Claim Participant data
        // ---------------------------------
        $submitterEmail = $submitter['email'];
        $emails = [];
        $lineCount = 0;
        foreach ($claimParticipants as $participant) {
            ++$lineCount;
            $participantEmail = $participant['participant_email'];
            if ($participantEmail == $submitterEmail) {
                $errors[] = sprintf("You cannot add yourself as an Additional Participant on line %s.\n", $lineCount);
            }
            if (! empty($emails[$participantEmail])) {
                $errors[] = sprintf("You cannot multiple of same Additional Participant %s on line %s.\n", $lineCount);
            }
        }

        if ((bool) setting('claim.use_simplified_lineitems') || (bool) setting('claim.offers.use_checkboxes')) {
            $formData = $this->transformSimplifiedLineItems($formData);
            $claimItems = $formData['claim_items'];
        }

        // ---------------------------------
        // Validate Claim Line Item data
        // ---------------------------------
        $maxLineItems = (int) setting('claim.max_lineitems');
        if (empty($errors)) {
            if ($maxLineItems > -1 && count($claimItems) < 1) {
                $errors[] = sprintf("You must enter at least 1 Claim Line Item.\n");
            }
        }

        if (empty($errors)) {
            if (count($claimItems) > 0) {
                // $errors = $this->claimRepository->validate('claim_lineitem', $claimItems);
                $errors = Form::validateFormData('claim_lineitem', $claimItems, $claimFormSettings);
                $quantityMin = setting('claim.lineitems.min_quantity', 1);
                $quantityMax = setting('claim.lineitems.max_quantity', 1000);
                // Eoghan: The following code should be refactored but for now
                // it will suffice. Eventually, we should handle form data with
                // multiple rows in a more dynamic, UI configured way.
                $validator = Validator::make(['claim_item' => $claimItems], [
                    'claim_item.*.offer_group' => 'required',
                    'claim_item.*.name' => 'required',
                    'claim_item.*.qty' => 'required|numeric|between:' . $quantityMin . ',' . $quantityMax
                ]);

                if ($validator->fails()) {
                    foreach ($validator->errors()->all() as $error) {
                        $errors[] = preg_replace_callback(
                            '/claim_item.([0-9]+).qty\ must be/',
                            function ($match) {
                                $index = (int) $match[0] + 1;
                                return "claim item at position $index for quantity must be";
                            },
                            $error
                        );
                    }
                }
            }
        }

        if (! empty($errors)) {
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        // ------------------------------
        // Populate the Form Fields with the Form Data
        // ------------------------------
        $review = false;
        $claimFields = Form::mergeFormFieldData('claim', $formData, $claimFormSettings, $review, $errors);

        if (! $claimFields) {
            if (empty($claimFields)) {
                $message = sprintf("The list of External Dynamic Claim Fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the External Dynamic Claim Field data.");
            }
            array_unshift($errors, $message);
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        // ------------------------------
        // Get Various Dropdown Options
        // ------------------------------
        $documentTypes          = $this->claimUtil->getDocumentTypes();
        $documentSuffixes       = $this->claimUtil->getDocumentSuffixes();

        // ---------------------------------
        // Re-display Data
        // ---------------------------------
        $pageData = [
            'mode'              => $mode,
            'code'              => $formData['code'],
            'claimStoreUri'     => '/claim/store',
            'claimFields'       => $claimFields,
            'formData'          => $formData,
            'documentTypes'     => $documentTypes,
            'documentSuffixes'  => $documentSuffixes,
            'uploadFiles'       => $uploadFiles,
        ];

        return view('claim.review', $pageData);
    }

    /**
     * Store a Claim.
     *
     * @return Response
     */
    public function store()
    {
        $data = request()->all();

        // ---------------------------------
        // Check if the Back Button was pressed
        // ---------------------------------
        $code           = $data['code'];
        $mode           = $data['mode'];
        $promotionId    = $data['offer_promotion_id'];
        $claimId        = $data['claim_id'];

        if ($mode == 'new') {
            $returnUri = '/claim/new/' . $promotionId . '/' . $code;
        } else {
            $returnUri = '/claim/edit/' . $claimId . '/' . $code;
        }

        if (isset($data['backButton']) && $data['backButton'] == 'Back') {
            return redirect($returnUri);
        }

        // -----------------------------------------
        // Get Form Data saved in session by code key
        // -----------------------------------------
        $formData = unserialize(session()->get($code));
        if (! $formData) {
            return $this->claimUtil->reEnterClaim('/claim/new');
        }

        if ((bool) setting('claim.use_simplified_lineitems') || (bool) setting('claim.offers.use_checkboxes')) {
            $formData = $this->transformSimplifiedLineItems($formData);
        }

        // -----------------------------------------
        // Create or Update Claim
        // -----------------------------------------
        $formData['class'] = 'claim';

        if ($mode == 'new') {
            $formData['action'] = 'createClaim';
            $response = $this->claimRepository->put($formData);
        } else {
            $formData['action'] = 'updateClaim';
            $response = $this->claimRepository->put($formData);
        }

        if (! $response->result) {
            $errors = $response->errors;
            $this->flash('error', $errors);
            return redirect($returnUri);
        }

        // -----------------------------------------
        // Clear Session data for this Claim
        // -----------------------------------------
        session()->forget($code);

        return redirect('/claim');
    }

    /**
     * Show the New Claim entry page.
     *
     * @param  string $promotionId
     * @param  string $code
     * @return Response
     */
    public function create($promotionId = '', $code = '')
    {
        $errors = [];
        $returnURI = '/claim/promotions';

        // Get Claim Submitter Participant Info
        $submitter = $this->claimUtil->getSubmitterParticipant($errors);

        if (! $submitter) {
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        // ------------------------------
        // Check for no Promotion Selected
        // ------------------------------

        $offerSvc = app()->make(\Ignite\Claim\Repositories\OfferRepository::class);

        if ($promotionId == '') {
            $response = $offerSvc->get([
                'class' => 'claim_offer_promotion',
                'action' => 'getList',
                'status' => 1
            ]);

            if (! $response->result) {
                $errors[] = sprintf("No Active Promotions.\n");
                $errors = array_merge($errors, $response->errors);
                $this->flash('error', $errors);
                return redirect($returnURI);
            }

            $promotions = $response->data;

            // If we have a Promotion count != 1, we need to go back to the Promotions list
            if (count($promotions) != 1) {
                if (count($promotions) == 0) {
                    $message = sprintf("No Active Promotions.\n");
                } else {
                    $message = sprintf("Please select a Promotion to make a claim.\n");
                }

                $this->flash('info', $message);
                return redirect($returnURI);
            }

            // If only 1 Promotion is Active we can stay here
            // and display the New Claim page for that Promotion
            $promotionId = $promotions[0]['id'];
        }

        // Check for New or Redisplay with Errors
        $promotionHtml  = '';
        $className      = 'claim';

        if ($code != '') {
            $formData = unserialize(session()->get($code));
            if (! $formData) {
                return $this->claimUtil->reEnterClaim($returnURI);
            }

            $filesCode = $code;
        } else {
            $code      = uniqid('');
            $filesCode = $code;
            $formData  = $this->claimRepository->toArray($className);
            $formData['claim_id']           = 'new';
            $formData['offer_promotion_id'] = $promotionId;
            $formData['claim_items']        = [];
            $formData['claim_participants'] = [];
            $formData['maxLineitems']       = setting('claim.max_lineitems', '-1');
            $formData['maxParticipants']    = setting('claim.max_participants');
        }

        // Get Form Fields
        $claimFormSettings = \Ignite\Claim\Entities\Form::findByKey(self::$claimFieldsExternalKey);
        if (! $claimFormSettings) {
            $errors[] = sprintf("Missing claim form configuration. Key = '%s'\n", self::$claimFieldsExternalKey);
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        // Populate the Form Fields with the Form Data
        $review = false;
        $claimFields = Form::mergeFormFieldData('claim', $formData, $claimFormSettings, $review, $errors);

        if (! $claimFields) {
            if (empty($claimFields)) {
                $message = sprintf("The list of External Dynamic Claim Fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the External Dynamic Claim Field data.");
            }
            array_unshift($errors, $message);
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        // Get any Uploaded Files from previous save
        $uploadFiles = (! empty($formData['upload_files'])) ? $formData['upload_files'] : [];

        // Get Promotion
        $response = $offerSvc->get([
            'class' => 'claim_offer_promotion',
            'action' => 'find',
            'id' => $promotionId
        ]);

        if (! $response->result) {
            $errors[] = sprintf("Can't find Promotion for Id = '%s'\n", $promotionId);
            $errors = array_merge($errors, $response->errors);
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        $promotion = $response->data;

        // Get Offer Groups/Offer
        $response = $offerSvc->get([
            'mode' => 'array',
            'class' => 'claim_offer',
            'action' => 'getOfferGroups',
            'status' => 1,
            'promotion_id' => $promotionId
        ]);
        $offerGroups = $response->data;

        if (! $offerGroups) {
            $errors[] = sprintf("Can't find Offers for Promotion Id = '%s'\n", $promotionId);
            $errors = array_merge($errors, $response->errors);
            $this->flash('error', $errors);
            return redirect($returnURI);
        }

        // Get Participant Types
        $participantTypes = $this->claimUtil->getParticipantTypes();
        $documentTypes    = $this->claimUtil->getDocumentTypes();
        $documentSuffixes = $this->claimUtil->getDocumentSuffixes();

        return view('claim.create', [
            'code'              => $code,
            'mode'              => 'new',
            'claimReviewUri'    => '/claim/review',
            'claimFields'       => $claimFields,
            'formData'          => $formData,
            'participantTypes'  => $participantTypes,
            'offerGroups'       => $offerGroups,
            'filesCode'         => $filesCode,
            'uploadFiles'       => $uploadFiles,
            'documentTypes'     => $documentTypes,
            'documentSuffixes'  => $documentSuffixes,
        ]);
    }

    /**
     * Show the Available Promotions page.
     *
     * @param int $campaignId
     *
     * @return Response
     */
    public function listPromotions($campaignId = 0)
    {
        $campaigns = $this->getAllActiveCampaigns();
        $promotions = $this->getActivePromotionsByCampaignId($campaignId);

        $pageData = [
            'campaignId' => $campaignId,
            'campaigns'  => $campaigns,
            'promotions' => $promotions,
        ];

        return view('claim.promotions.index', $pageData);
    }

    private function getActivePromotionsByCampaignId($campaignId)
    {
        // @todo Replace with query scope for status called onlyActive()
        // OfferPromotion::onlyActive()->byCampaign($campaignId)->orderBy('name')->get();
        $query = OfferPromotion::where('status', '=', 1);

        if ($campaignId > 0) {
            $query->where('campaign_id', '=', $campaignId);
        }

        return $query->orderBy('name')->get()->toArray();
    }

    private function getAllActiveCampaigns()
    {
        // @todo Replace with query scope for status called onlyActive()
        //return OfferCampaign::onlyActive()->orderBy('name')->get()->toArray();
        return OfferCampaign::where('status', '=', 1)->orderBy('name')->get()->toArray();
    }

    public function start($promotionId = '', $code = '')
    {
        return view('claim.new', []);
    }

    /**
     * In the case of a simplified UI for adding line items, we need transform the
     * lineitems for use in the Rule engine.
     *
     * @param  array $formData
     * @return array
     */
    private function transformSimplifiedLineItems($formData)
    {
        $firstItem = $formData['claim_items'][0];
        if (strpos($firstItem['offer_id'], ',') !== false && strpos($firstItem['name'], '|') !== false) {
            $items = array_combine(
                preg_split('/\s?,\s?/', $firstItem['offer_id']),
                preg_split('/\s?\|\s?/', $firstItem['name'])
            );
            $lineItems = [];
            foreach ($items as $id => $name) {
                $lineItem = $firstItem;
                $lineItem['name'] = $name;
                $lineItem['offer_id'] = $id;
                $lineItems[] = $lineItem;
            }
            $formData['claim_items'] = $lineItems;
        }

        return $formData;
    }
}
