<?php

namespace Ignite\Claim\Repositories;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Ignite\Claim\Entities\RuleLog;
use Ignite\Claim\Entities\File;
use Ignite\Claim\Entities\ClaimParticipant;
use Ignite\Claim\Events\ClaimEvent;
use Ignite\Claim\Models\CapCheck;
use Ignite\Core\Repositories\ParticipantRepository;
use Ignite\Core\Repositories\TransactionRepository;

class ClaimRepository extends CoreRepository
{
    /** @var TransactionRepository */
    private $transactionRepository;

    /** @var RuleRepository */
    private $ruleRepository;

    /**
     * Allowed filters for the claim data.
     *
     * @var array
     */
    public static $claimFilters = [
        'user_id' => ['type' => 'integer'],
        'claim_type' => ['type' => 'string'],
    ];

    /**
     * Allowed filters for the claim line item data.
     *
     * @var array
     */
    public static $claimLineItemFilters = [
        'claim_id' => ['type' => 'integer'],
        'claim_offer_id' => ['type' => 'integer'],
        'name' => ['type' => 'string'],
        'offer_group' => ['type' => 'string'],
        'status' => ['type' => 'integer'],
    ];

    /**
     * Filters for the claim participant data.
     *
     * @var array
     */
    public static $claimParticipantFilters = [
        'claim_id' => ['type' => 'integer'],
        'user_id' => ['type' => 'integer'],
        'participant_type' => ['type' => 'string'],
        'participant_email' => ['type' => 'string'],
        'date_approved' => ['type' => 'date'],
        'date_canceled' => ['type' => 'date'],
        'date_declined' => ['type' => 'date'],
        'date_issued' => ['type' => 'date'],
        'date_submitted' => ['type' => 'date'],
        'status' => ['type' => 'string'],
    ];

    /**
     * ClaimRepository constructor.
     *
     * @param TransactionRepository $transactionRepository
     * @param RuleRepository $ruleRepository
     */
    public function __construct(TransactionRepository $transactionRepository, RuleRepository $ruleRepository)
    {
        $this->transactionRepository = $transactionRepository;
        $this->ruleRepository = $ruleRepository;

        $classes = [
            [
                'path' => 'Ignite\Claim\Entities\Claim',
                'filters' => self::$claimFilters,
                'defaultTypesKey' => 'module.claim.types.*',
            ],
            [
                'path' => 'Ignite\Claim\Entities\ClaimLineitem',
                'filters' => self::$claimLineItemFilters,
            ],
            [
                'path' => 'Ignite\Claim\Entities\ClaimParticipant',
                'filters' => self::$claimParticipantFilters,
            ],
        ];

        parent::__construct($classes);
    }

    /**
     * Get data from the database.
     *
     * @param  array $params
     * @return bool
     */
    public function get($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['action', 'class'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $action = $params['action'];

        switch ($action) {
            default:
                parent::get($params);
                break;
        }

        return $this->response;
    }

    public function put($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['action', 'class'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $action = $params['action'];

        switch ($action) {
            case 'createClaim':
                $rc = $this->createClaim($params);
                break;

            case 'updateClaim':
                $rc = $this->updateClaim($params);
                break;

            case 'deleteClaim':
                $rc = $this->deleteClaim($params);
                break;

            case 'approveClaim':
                $params['status'] = 'approved';
                $rc = $this->updateStatus($params);
                break;

            case 'declineClaim':
                $params['status'] = 'declined';
                $rc = $this->updateStatus($params);
                break;

            case 'changeStatus':
                $rc = $this->updateStatus($params);
                break;

            default:
                $this->response->result = false;
                $this->response->errors[] = sprintf("%s: Unknown action param '%s'.", $function, $action);
                break;
        }

        return $this->response;
    }

    private function createClaim($formData)
    {
        $function = __FUNCTION__;

        // -----------------------------------------
        // Create New Claim
        // -----------------------------------------
        DB::beginTransaction();

        $claimData = $formData;
        $claimData['class'] = 'claim';
        $claimData['status'] = 'pending';

        if (isset($formData['user_id'])) {
            $claimData['user_id'] = $formData['user_id'];
        }

        if (! isset($claimData['user_id'])) {
            $claimData['user_id'] = auth()->user()->getKey();
        }

        if (! array_key_exists('date_submitted', $claimData)) {
            $claimData['date_submitted'] = now();
        }

        // -------------
        // Need to unset created_at, in case it was passed in, that way we get the current Date/Time.
        // This would be because it was needed for the Rule Calculations
        // -------------
        unset($claimData['created_at']);

        $response = parent::create($claimData);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Unable to Create New Claim record.");
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        $claim   = $response->data;
        $claimId = $claim['id'];

        // -----------------------------------------
        // Create New Claim Line Items
        // -----------------------------------------
        //$maxLineItems = (int) setting('claim.max_lineitems'); @todo determine whether we can remove this
        $claimItems = $formData['claim_items'];

        if (count($claimItems) > 0) {
            foreach ($claimItems as $index => $tmpClaimItem) {
                $claimItem             = &$claimItems[$index];
                $claimItem['class']    = 'claim_lineitem';
                $claimItem['claim_id'] = $claimId;
                $claimItem['status']   = 1;
            }

            $response = parent::create($claimItems);

            if (! $response->result) {
                DB::rollBack();
                $message = sprintf("Unable to Create New Claim Item records.");
                array_unshift($response->errors, $message);
                $this->response = $response;

                return false;
            }
        } else {
            $claimItems = [];
        }

        // -----------------------------------------
        // Use the Submitting Participant the first Claim Participant
        // -----------------------------------------
        $userId = $claimData['user_id'];
        $participant = $this->getParticipantByUserId($userId);

        if (! $participant) {
            DB::rollBack();
            $response->result = false;
            $message = sprintf("No Participant record found for Current User Id %s.\n", $userId);
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        $claimParticipants   = [];
        $claimParticipants[] = $this->buildClaimParticipant($participant);

        // Create New Claim Participants
        if (! empty($formData['claim_participants'])) {
            $claimParticipants = array_merge($claimParticipants, $formData['claim_participants']);
        }

        foreach ($claimParticipants as $index => $tmpClaimParticipant) {
            if (! filter_var($tmpClaimParticipant['participant_email'], FILTER_VALIDATE_EMAIL)) {
                $response->result = false;
                $message = sprintf("Participant email is invalid: %s.\n", $tmpClaimParticipant['participant_email']);
                array_unshift($response->errors, $message);
                $this->response = $response;
                return false;
            }

            // Check if this Claim Participant is already Registered
            // if so use that Participant Info
            $registered = false;
            $participant = $this->getParticipantByEmail($tmpClaimParticipant['participant_email']);

            if ($participant && isset($claimParticipants[$index]['participant_type'])) {
                $participant['type'] = $claimParticipants[$index]['participant_type'];
            }

            if ($participant) {
                $claimParticipants[$index] = $this->buildClaimParticipant($participant);
                $registered = true;
            }

            $claimParticipant = &$claimParticipants[$index];

            $claimParticipant['class'] = 'claim_participant';
            $claimParticipant['claim_id'] = $claimId;
            if ($registered) {
                $claimParticipant['status'] = 'pending';
            } else {
                $claimParticipant['status'] = 'waiting';
            }

            if (! array_key_exists('date_submitted', $claimParticipant)) {
                $claimParticipant['date_submitted'] = now();
            }

            // Calculate the Participant Claim values for this Claim
            $calcParams                         = [];
            $calcParams['class']                = 'claim_participant';  // <- Not really needed, but required by the Interface
            $calcParams['action']               = 'getClaimValue';
            $calcParams['claim']                = $claim;
            $calcParams['claim']['participant'] = $claimParticipant;
            $calcParams['claim']['lineitems']   = $claimItems;
            $calcParams['participantType']      = $claimParticipant['participant_type'];

            $ruleResponse = $this->ruleRepository->get($calcParams);

            if (! $ruleResponse->result) {
                DB::rollBack();
                $message = sprintf(
                    "Unable to calculate claim value for participant with email: '%s'.",
                    $claimParticipant['participant_email']
                );
                array_unshift($ruleResponse->errors, $message);
                $this->response = $ruleResponse;

                return false;
            }

            $logs   = $ruleResponse->data->logs;
            $values = $ruleResponse->data->values;

            $claimParticipant['value_adjust']     = 0;
            $claimParticipant['value_calculated'] = $values['claim_total'];
            $claimParticipant['value']            = $values['claim_total'];

            if ($claimParticipant['value'] == 0 && ! setting('claim.allow_zero_value', false)) {
                DB::rollBack();
                $response->result = false;
                $message = sprintf(
                    "Claim would result in zero value for participant with email: '%s'.",
                    $claimParticipant['participant_email']
                );
                array_unshift($response->errors, $message);
                $this->response = $response;

                return false;
            }

            // -----------------------------------------
            // Create New Claim Participant record so we have an ID for the Rule Log record
            // -----------------------------------------
            $response = parent::create($claimParticipant);

            if (! $response->result) {
                DB::rollBack();
                $message = sprintf("Unable to create new claim participant record.");
                array_unshift($response->errors, $message);
                $this->response = $response;

                return false;
            }

            $newClaimParticipant = $response->data;

            // -----------------------------------------
            // Create New Rule Log record for this Claim Participant
            // -----------------------------------------
            $ruleLog                         = [];
            $ruleLog['class']                = 'claim_rule_log';
            $ruleLog['claim_participant_id'] = $newClaimParticipant['id'];
            $ruleLog['log_text']             = $this->buildRuleLogText($values, $logs);

            /** @var RuleLog $response */
            $response = $this->ruleRepository->create($ruleLog);

            if (! $response->result) {
                DB::rollBack();
                $message = sprintf("Unable to create new rule log record.");
                array_unshift($response->errors, $message);
                $this->response = $response;
                return false;
            }
        }

        // -----------------------------------------
        // Create the Uploaded File records, and move files to Permanent Location
        // -----------------------------------------
        if (! $this->processDocumentFiles($claimId, $formData, $message)) {
            DB::rollBack();
            $this->response->result = false;
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // -----------------------------------------
        // Commit the changes to the Database
        // -----------------------------------------
        DB::commit();

        // ---------------------------------
        // Get Data needed for Event Handlers
        // ---------------------------------
        $response = parent::get([
            'id' => $claimId,
            'class' => 'claim',
            'action' => 'find',
            'mode' => 'array',
            'with' => [
                'participant',
                'claim_participants',
                'claim_participants.participant'
            ],
        ]);
        $claimEventData = $response->data;

        $tmpClaimParticipants = $claimEventData['claim_participants'];

        // ---------------------------------
        // Generate Events for Post Processing, like emails.
        // ---------------------------------
        foreach ($tmpClaimParticipants as $index => $tmpClaimParticipant) {
            $claimEventData['email_participant'] = $tmpClaimParticipant;

            if ($tmpClaimParticipant['user_id'] == $claim['user_id']) {
                $eventType = 'claim_submitted';
            } else {
                $eventType = 'claim_submitted_split';
            }

            event(new ClaimEvent($eventType, $claimEventData));
        }

        event(new \Ignite\Claim\Events\ClaimCreated((int) $claimId));

        return true;
    }

    private function deleteClaim($formData)
    {
        $function = __FUNCTION__;

        $requireds = ['claim_id'];
        if (! $this->checkRequireds($function, $formData, $requireds)) {
            return false;
        }

        // ---------------------------------
        // Get Claim Data from Database
        // ---------------------------------
        $claimId = $formData['claim_id'];

        $params = [
            'class' => 'claim',
            'action' => 'find',
            'mode' => 'array',
            'with' => [
                'lineItems',
                'claim_participants',
                'claim_participants.rule_log',
                'claim_participants.transactions'
            ],
            'id' => $claimId
        ];

        $response = parent::get($params);

        if (! $response->result) {
            $message = sprintf("Can't find Claim for Id = '%s'\n", $claimId);
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        $claim             = $response->data;
        $claimItems        = (! empty($claim['line_items'])) ? $claim['line_items'] : [];
        $claimParticipants = (! empty($claim['claim_participants'])) ? $claim['claim_participants'] : [];

        // ---------------------------------
        // Delete Claim Participant(s)
        // ---------------------------------
        DB::beginTransaction();

        if (count($claimParticipants) > 0) {
            $deleteParticipants = [];
            foreach ($claimParticipants as $index => $tmpClaimParticipant) {
                if (! empty($formData['claim_participant_id']) && $tmpClaimParticipant['id'] != $formData['claim_participant_id']) {
                    continue;
                }

                // /////////////////////////////////////////////////
                // /////////////////////////////////////////////////
                // IMPORTANT: TODO:
                // WE NEED TO DELETE Rule Log and any Transactions for this Claim / Participant
                // /////////////////////////////////////////////////
                // /////////////////////////////////////////////////

                // ---------------------------------
                // Delete Rule Log record
                // ---------------------------------

                // ---------------------------------
                // Delete Transaction record(s)
                // ---------------------------------

                $deleteParticipant          = $tmpClaimParticipant;
                $deleteParticipant['class'] = 'claim_participant';

                $deleteParticipants[] = $deleteParticipant;
                unset($claimParticipants[$index]);
            }

            if (count($deleteParticipants) === 1) {
                $response = parent::delete($deleteParticipants);

                if (! $response->result) {
                    DB::rollBack();
                    $message = sprintf("Problem Deleting Claim Participant(s) for Claim Id = '%s'\n", $claimId);
                    array_unshift($response->errors, $message);
                    $this->response = $response;

                    return false;
                }
            }
        }

        // ---------------------------------
        // If we were only deleting a Single Claim Participant
        // and there are still Claim Participants, we are done
        // ---------------------------------
        if (! empty($formData['claim_participant_id']) && count($claimParticipants) > 0) {
            DB::commit();

            return true;
        }

        // ---------------------------------
        // Delete Lineitems for Claim
        // ---------------------------------
        if (is_array($claimItems) && count($claimItems) > 0) {
            foreach ($claimItems as $index => $tmpClaimItem) {
                $claimItem          = &$claimItems[$index];
                $claimItem['class'] = 'claim_lineitem';
            }

            $response = parent::delete($claimItems);

            if (! $response->result) {
                DB::rollBack();
                $message = sprintf("Problem Deleting Lineitems for Claim Id = '%s'\n", $claimId);
                array_unshift($response->errors, $message);
                $this->response = $response;

                return false;
            }
        }

        // ---------------------------------
        // Delete Claim
        // ---------------------------------
        $claim['class'] = 'claim';

        $response = parent::delete($claim);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Could not Delete Claim for Id = '%s'\n", $claimId);
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        return true;
    }

    private function updateClaim($formData)
    {
        $function  = __FUNCTION__;
        $requireds = ['claim_id'];
        if (! $this->checkRequireds($function, $formData, $requireds)) {
            return false;
        }

        $skipRuleCalculation = config('claim.skip_rule_calculation_on_update', false);
        if (isset($formData['skip_rule_calculation']) && (int) $formData['skip_rule_calculation'] === 1) {
            $skipRuleCalculation = true;
            unset($formData['skip_rule_calculation']);
        }

        $claimId = $formData['claim_id'];

        // -----------------------------------------
        // Get Existing Claim data
        // -----------------------------------------
        $params = [
            'class' => 'claim',
            'action' => 'find',
            'mode' => 'array',
            'with' => [
                'lineItems',
                'claim_participants'
            ],
            'id' => $claimId
        ];

        $response = parent::get($params);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Can't find Existing Claim for Id = '%s'\n", $claimId);
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        $currentClaim = $response->data;
        $currentItems = (! empty($currentClaim['line_items'])) ? $currentClaim['line_items'] : [];

        // ---------------------------------
        // Get Specific Claim Participant we are updating
        // ---------------------------------
        $claimParticipantId = $formData['claim_participant']['id'];
        $claimParticipant   = false;
        foreach ($currentClaim['claim_participants'] as $participant) {
            if ($participant['id'] == $claimParticipantId) {
                $claimParticipant = $participant;
            }
        }

        if (! $claimParticipant) {
            DB::rollBack();
            $response->result = false;
            $message          = sprintf(
                "Could not find Claim Participant %s for Claim %s.",
                $claimParticipantId,
                $claimId
            );
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // -----------------------------------------
        // Update Claim data
        // -----------------------------------------
        $formData['class'] = 'claim';

        $claimData = parent::data($formData);

        $claimData['id']    = $claimId;
        $claimData['class'] = 'claim';

        DB::beginTransaction();

        $response = parent::update($claimData);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Unable to Update Claim record for Claim Id = '%s'\n", $claimId);
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // ---------------------------------
        // Update Line Item Records for Claim
        // ---------------------------------
        $class      = 'claim_lineitem';
        $items      = $formData['claim_items'];
        $claimItems = $formData['claim_items'];

        $foreignKeyName  = 'claim_id';
        $foreignKeyValue = $claimId;

        if (count($items) > 0) {
            foreach ($items as $index => $tmpItem) {
                $item             = &$items[$index];
                $item['class']    = (empty($item['class'])) ? $class : $item['class'];
                $item['claim_id'] = (empty($item['claim_id'])) ? $claimId : $item['offer_id'];
            }
        }

        $response = parent::sync($items, $class, $foreignKeyName, $foreignKeyValue);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Problem Updating '%s' Child records for Claim Id = '%s'\n", $class, $claimId);
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // -----------------------------------------
        // Recalculate the Claim Participant's Claim Value
        // -----------------------------------------
        $ruleSvc = app()->make(\Ignite\Claim\Repositories\RuleRepository::class);

        $calcParams = [];
        $calcParams['class']                = 'claim_participant';  // <- Not really needed, but required by the Interface
        $calcParams['action']               = 'getClaimValue';
        $calcParams['claim']                = $claimData;
        $calcParams['claim']['participant'] = $claimParticipant;
        $calcParams['claim']['lineitems']   = $claimItems;
        $calcParams['participantType']      = $claimParticipant['participant_type'];

        $ruleResponse = $ruleSvc->get($calcParams);
        $values       = $ruleResponse->data->values;
        $logs         = $ruleResponse->data->logs;

        $claimValue = $values['claim_total'];

        if ($claimValue == 0 && ! setting('claim.allow_zero_value', false)) {
            DB::rollBack();
            $response->result = false;
            $message = sprintf(
                "Claim would result in Zero Value for Participant with email: '%s'.",
                $claimParticipant['participant_email']
            );
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // -----------------------------------------
        // Update the Claim Participant's data
        // -----------------------------------------
        $claimParticipantData = $formData['claim_participant'];
        $claimParticipantData['id'] = $claimParticipant['id'];
        $claimParticipantData['class'] = 'claim_participant';

        if (! $skipRuleCalculation) {
            $claimParticipantData['value_calculated'] = $claimValue;
            // Handle case where value_adjust is not on the Claim Edit Form
            $valueAdjust = (isset($claimParticipantData['value_adjust'])) ? $claimParticipantData['value_adjust'] : 0;
            $claimParticipantData['value'] = $claimValue + $valueAdjust;
        }

        $response = parent::update($claimParticipantData);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf(
                "Unable to Update Claim Participant Id = '%s'record, for Claim Id = '%s'\n",
                $claimParticipant['id'],
                $claimId
            );
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // -----------------------------------------
        // Try to find an existing Rule Log for the Claim Participant
        // -----------------------------------------
        $params = [
            'class' => 'claim_rule_log',
            'action' => 'getList',
            'mode' => 'array',
            'claim_participant_id' => $claimParticipantId
        ];

        $response = $ruleSvc->get($params);

        if ($response->result && is_array($response->data) && isset($response->data[0])) {
            $foundLog = true;
            $ruleLog = $response->data[0];
        } else {
            $foundLog = false;
            $ruleLog = [];
            $ruleLog['claim_participant_id'] = $claimParticipantData['id'];
        }

        $ruleLog['class'] = 'claim_rule_log';
        $ruleLog['log_text'] = $this->buildRuleLogText($values, $logs);

        if ($foundLog) {
            $response = $ruleSvc->update($ruleLog);
        } else {
            $response = $ruleSvc->create($ruleLog);
        }

        if (! $response->result) {
            DB::rollBack();
            if ($foundLog) {
                $message = sprintf("Unable to Update Rule Log record.");
            } else {
                $message = sprintf("Unable to Create New Rule Log record.");
            }
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // -----------------------------------------
        // Add / Delete the Uploaded File records, and move files to Permanent Location
        // -----------------------------------------
        if (! $this->processDocumentFiles($claimId, $formData, $message)) {
            DB::rollBack();
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        event(new \Ignite\Claim\Events\ClaimUpdated((int) $claimId));

        return true;
    }

    /**
     * Given an array of form data, update the status for a claim participant.
     *
     * @param array $formData
     *
     * @return bool
     */
    private function updateStatus($formData)
    {
        $function  = __FUNCTION__;
        $requireds = ['claim_participant_id', 'status'];
        if (! $this->checkRequireds($function, $formData, $requireds)) {
            return false;
        }

        // ---------------------------------
        // Get Claim Participant from Database
        // ---------------------------------
        $claimParticipantId = $formData['claim_participant_id'];
        $params = [
            'class' => 'claim_participant',
            'action' => 'find',
            'mode' => 'array',
            'with' => ['participant'],
            'id' => $claimParticipantId
        ];

        $response = parent::get($params);

        if (! $response->result) {
            $message = sprintf("Can't find Claim Participant for Id = '%s'\n", $claimParticipantId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            throw new \Exception($message);
            return false;
        }

        $claimParticipant = $response->data;

        // ---------------------------------
        // Determine what needs to be updated for the new Status
        // ---------------------------------
        $oldStatus   = $claimParticipant['status'];
        $status      = $formData['status'];
        $foundStatus = false;
        $dateNow     = date('Y-m-d H:i:s');

        switch ($status) {
            case 'approved':
                $claimParticipant['date_approved'] = $dateNow;
                $foundStatus                       = true;
                break;

            case 'cancelled':
                $claimParticipant['date_canceled']   = $dateNow;
                $claimParticipant['reason_declined'] = $formData['reason_declined'];
                $foundStatus                         = true;
                break;

            case 'declined':
                $claimParticipant['date_declined']   = $dateNow;
                $claimParticipant['reason_declined'] = $formData['reason_declined'];
                $foundStatus                         = true;
                break;

            case 'issued':
                $claimParticipant['date_issued'] = $dateNow;
                $foundStatus                     = true;
                break;

            default:
                break;
        }

        if (! $foundStatus) {
            $message = sprintf(
                "Unknown Status '%s', Claim Participant for Id = '%s', NOT updated.\n",
                $status,
                $claimParticipantId
            );
            throw new \Exception($message);
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // ---------------------------------
        // Update Claim Participant on Database
        // ---------------------------------
        DB::beginTransaction();

        $claimParticipant['class']  = 'claim_participant';
        $claimParticipant['status'] = $status;

        $response = parent::update($claimParticipant);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Unable to Update Claim Participant for Id = '%s'\n", $claimParticipantId);

            throw new \Exception($message);
            array_unshift($response->errors, $message);
            $this->response = $response;

            return false;
        }

        // ---------------------------------
        // For Approvals, Add Transaction for Claim
        // ---------------------------------
        if ($status == 'approved') {
            // ---------------------------------------
            // Check if this Claim would be denied due to a Promotion Cap
            // ---------------------------------------
            //$capCheck = new CapCheck($this, new OfferRepository(), new TransactionRepository());
            $capCheck = app(CapCheck::class, ['claimRepository' => $this]);
            $response = $capCheck->checkCap($claimParticipantId);

            if (! $response->result) {
                DB::rollBack();
                $this->response = $response;
                return false;
            }

            try {
                $transaction = $this->transactionRepository->create([
                    'user_id'      => $claimParticipant['user_id'],
                    'description'      => "Claim # " . $claimParticipant['claim_id'],
                    'transaction_date' => $dateNow,
                    'type'             => 'EARNED',
                    'related_name'     => 'CLAIM',
                    'related_id'       => $claimParticipant['id'],
                    'related_type'     => ClaimParticipant::class,
                    'value'            => $claimParticipant['value'],
                ]);
            } catch (\Exception $e) {
                DB::rollBack();
                array_unshift($response->errors, sprintf(
                    "Unable to create transaction record during claim approval for participant with Id: %s",
                    $claimParticipantId
                ));
                $this->response = $response;
                return false;
            }
        }

        if ($status === 'cancelled') {
            try {
                $transaction = $this->transactionRepository->create([
                    'user_id'          => $claimParticipant['user_id'],
                    'description'      => "Claim # " . $claimParticipant['claim_id'],
                    'transaction_date' => $dateNow,
                    'type'             => 'CANCELLED',
                    'related_name'     => 'CLAIM',
                    'related_id'       => $claimParticipant['id'],
                    'related_type'     => ClaimParticipant::class,
                    'value'            => -$claimParticipant['value'],
                ]);
            } catch (\Exception $e) {
                DB::rollBack();
                array_unshift($response->errors, sprintf(
                    "Unable to create transaction record during claim cancellation for participant with Id: %s",
                    $claimParticipantId
                ));
                $this->response = $response;
                return false;
            }
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        // ---------------------------------
        // Check if we actually made a change
        // ---------------------------------
        if (! $foundStatus) {
            throw new \Exception('No change');
            return true;
        }

        // ---------------------------------
        // Get Data needed for Event Handlers
        // ---------------------------------
        $params = [
            'class' => 'claim',
            'action' => 'find',
            'mode' => 'array',
            'with' => [
                'participant',
                'claim_participants',
                'claim_participants.participant'
            ],
            'id' => $claimParticipant['claim_id']
        ];

        $response = parent::get($params);

        $claimEventData = $response->data;

        // Generate Event for Post Processing, like emails.
        $claimEventData['email_participant'] = $claimParticipant;

        event(new ClaimEvent('claim_' . $status, $claimEventData));

        return true;
    }

    private function getParticipantByUserId($userId)
    {
        return app()->make(ParticipantRepository::class)->find($userId);
    }

    private function getParticipantByEmail($email)
    {
        return app()->make(ParticipantRepository::class)->findByEmail($email);
    }

    private function buildClaimParticipant($participant)
    {
        $claimParticipant = [];

        $claimParticipant['user_id']       = $participant['user_id'];
        $claimParticipant['participant_email'] = $participant['email'];
        $claimParticipant['participant_type']  = $participant['type'];

        return $claimParticipant;
    }

    private function buildRuleLogText($values, $logs)
    {
        $ruleLogText = '';

        foreach ($values as $lineItem => $rules) {
            if ($lineItem == 'claim_total') {
                continue;
            }

            $ruleLogText .= sprintf("%s\n", $lineItem);

            foreach ($rules as $rule => $value) {
                $ruleLogText .= sprintf("    %s: %s\n", $rule, $value);
            }
        }

        if (isset($values['claim_total'])) {
            $ruleLogText .= sprintf("claim_total: %s\n", $values['claim_total']);
        }

        $ruleLogText .= "\n=================================================\n";

        $logText = '';
        foreach ($logs as $log) {
            $logText .= sprintf("(%s) %s\n", $log['level'], $log['message']);
        }

        $ruleLogText .= $logText;

        return $ruleLogText;
    }

    private function processDocumentFiles($claimId, $formData, &$message)
    {
        // -----------------------------------------
        // Make sure we know where to Get and Put the files
        // -----------------------------------------
        $claimPermanentPathKey = 'claim.file.path';
        $claimTemporaryPathKey = 'claim.file.temp_path';

        $claimFileTemporaryPath = setting($claimTemporaryPathKey);
        $claimFilePermanentPath = setting($claimPermanentPathKey);

        if (empty($claimPermanentPathKey)) {
            $message = sprintf("No file path for Path Setting '%s'", $claimPermanentPathKey);
            return false;
        }

        if (empty($claimFilePermanentPath)) {
            $message = sprintf("No file path for Path Setting '%s'", $claimTemporaryPathKey);
            return false;
        }

        $uploadedFiles = (! empty($formData['upload_files'])) ? $formData['upload_files'] : [];

        // -----------------------------------------
        // Get List of Existing Files / Ids
        // -----------------------------------------
        $existingFiles = File::where('related_id', '=', $claimId)
                             ->where('related_name', '=', 'claim')
                             ->where('related_type', '=', 'table')
                             ->get();

        $serial      = 0;
        $existingIds = [];
        if (count($existingFiles) > 0) {
            foreach ($existingFiles as $existingFile) {
                $existingIds[] = $existingFile->id;
                $parts         = explode('_', $existingFile->file_name);
                $serial        = ($parts[1] > $serial) ? $parts[1] : $serial;
            }
        }

        // -----------------------------------------
        // Create the New Uploaded File records, and move Files to Permanent Location
        // -----------------------------------------
        $updatedIds = [];
        foreach ($uploadedFiles as $uploadedFile) {
            // -------------------------
            // We only have to process New Uploads, as old ones can only be Deleted
            // -------------------------
            if ($uploadedFile['fileId'] > 0) {
                $updatedIds[] = $uploadedFile['fileId'];
                continue;
            }

            // -------------------------
            // Build correct Names for files
            // -------------------------
            ++$serial;
            $originalFileName = $uploadedFile['fileName'];
            $originalFileExt  = pathinfo($uploadedFile['fileName'], PATHINFO_EXTENSION);

            $permanentFileName = str_replace('claim_' . $formData['filesCode'] . '_', '', $uploadedFile['storeFileName']);
            $permanentFileName = $claimId . '_' . $serial . '_' . $permanentFileName;

            // -------------------------
            // Create the Uploaded File record for New Uploads
            // -------------------------
            $file = new File;

            $file->location_type         = 'local';
            $file->location_path_setting = $claimPermanentPathKey;

            $file->related_type = 'table';
            $file->related_name = 'claim';
            $file->related_id   = $claimId;

            $file->original_name = $uploadedFile['fileName'];
            $file->original_ext  = $originalFileExt;
            $file->file_name     = $permanentFileName;
            $file->file_ext      = $originalFileExt;

            $file->tags              = $uploadedFile['documentType'];
            $file->short_description = (! empty($uploadedFile['shortDescription'])) ? $uploadedFile['shortDescription'] : '';
            $file->description       = (! empty($uploadedFile['description'])) ? $uploadedFile['description'] : '';

            $file->user_id = auth()->user()->getKey();
            $file->login_user_id = app('impersonation')->getImpersonatorId() ?: auth()->user()->getKey();

            $storeToDatabase = $file->save();

            if (! $storeToDatabase) {
                $message = sprintf("Unable to Create New File Upload record.");
                return false;
            }

            // Move the Uploaded File to its permanent location
            $temporaryFullPath = $claimFileTemporaryPath . '/' . $uploadedFile['storeFileName'];
            $permanentFullPath = $claimFilePermanentPath . '/' . $file->file_name;
            $storeToDisk = Storage::disk(config('claim.disk'))->move($temporaryFullPath, $permanentFullPath);

            if (! $storeToDisk) {
                $message = sprintf("Unable to Move Upload file '%s'.", $uploadedFile['fileName']);
                return false;
            }
        }

        // Remove the Deleted File Records
        foreach ($existingFiles as $existingFile) {
            if (in_array($existingFile->id, $updatedIds)) {
                continue;
            }

            $permanentFullPath = $claimFilePermanentPath . '/' . $existingFile->file_name;
            $deleteFromDisk = Storage::disk(config('claim.disk'))->delete($permanentFullPath);

            if (! $deleteFromDisk) {
                $message = sprintf("Unable to remove deleted file '%s' from disk.", $existingFile->file_name);
                return false;
            }

            $deleteFromDatabase = $existingFile->delete();
            if (! $deleteFromDatabase) {
                $message = sprintf("Unable to delete file record for '%s'.", $existingFile->file_name);
                return false;
            }
        }

        return true;
    }
}
