<?php

namespace Ignite\Claim\Models;

use Ignite\Claim\Entities\Setting;
use Ignite\Core\Contracts\Repositories\ParticipantRepository;

class ClaimUtil
{
    protected $participantRepository;

    public function __construct(ParticipantRepository $participantRepository)
    {
        $this->participantRepository = $participantRepository;
    }

    /**
     * Get the claim participant types from the settings first then defer to
     * finding the hard-coded types.
     *
     * @return array
     */
    public function getParticipantTypes()
    {
        if ($participantTypes = setting('claim.participant_types')) {
            $types = explode("\n", $participantTypes);
            return array_combine($types, $types);
        }

        return $this->participantRepository->getTypes();
    }

    public function getDocumentTypes()
    {
        $documentTypes = [];

        $value = setting('claim.document.requires', '');
        $docTypes = preg_split('/[|\n]/', $value);

        foreach ($docTypes as $docType) {
            $parts = explode(':', $docType);
            $documentType = [];
            $documentType['name'] = $parts[0];
            $documentType['min'] = (isset($parts[1])) ? $parts[1] : '0';
            $documentType['max'] = (isset($parts[2])) ? $parts[2] : '10';

            // must have name and max > 0
            // @todo There's an implicit interface here, extract to an explicit interface?
            if ('' != $documentType['name'] && $documentType['max'] > 0) {
                $documentTypes[] = $documentType;
            }
        }

        return $documentTypes;
    }

    public function getDocumentSuffixes()
    {
        $value = setting('claim.document.suffixes', '');
        $documentSuffixes = [];
        $docSuffixes = preg_split('/[|\n]/', $value);
        foreach ($docSuffixes as $docSuffix) {
            $documentSuffixes[$docSuffix] = strtolower(str_replace('.', '', trim($docSuffix)));
        }

        return $documentSuffixes;
    }

    public function filterByStatus($claims, $status)
    {
        foreach ($claims as $claimIndex => $claim) {
            foreach ($claim['claim_participants'] as $participantIndex => $participant) {
                if ($participant['status'] != $status) {
                    unset($claims[$claimIndex]['claim_participants'][$participantIndex]);
                }
            }
            if (count($claims[$claimIndex]['claim_participants']) == 0) {
                unset($claims[$claimIndex]);
            }
        }

        return $claims;
    }

    public function reEnterClaim($returnUri)
    {
        $errors = [];

        $errors[] = sprintf("Missing Data for Session Token.\n");
        $errors[] = sprintf("Probable timeout.\n");
        $errors[] = sprintf("Please re-enter your Claim.\n");

        return redirect($returnUri)->with('errors', $errors);
    }

    public function getSubmitterParticipant(&$errors)
    {
        $userId = auth()->user()->getKey();

        try {
            return $this->participantRepository->find($userId);
        } catch (\Exception $e) {
            $errors[] = sprintf("No Participant record found for Current User Id %s.\n", $userId);
            $errors[] = sprintf("Claims cannot be entered without a Participant record.\n");
            return false;
        }
    }

    public function convertDocumentsToUploadFiles($documents)
    {
        $uploadFiles = [];

        foreach ($documents as $document) {
            $uploadFiles[] = [
                'fileId' => $document['id'],
                'documentType' => $document['tags'],
                'fileName' => $document['original_name'],
                'storeFileName' => $document['file_name'],
            ];
        }

        return $uploadFiles;
    }
}
