<?php

namespace Ignite\Claim\Models;

use Ignite\Claim\Repositories\ClaimRepository;
use Ignite\Claim\Entities\OfferPromotionCap;
use Ignite\Claim\Repositories\OfferRepository;
use Ignite\Core\Repositories\TransactionRepository;

class CapCheck
{
    /** @var ClaimRepository */
    protected $claimRepository;

    /** @var OfferRepository */
    protected $offerRepository;

    /** @var TransactionRepository */
    protected $transactionRepository;

    /**
     * Create a new CapCheck instance.
     *
     * @param ClaimRepository $claimRepository
     * @param OfferRepository $offerRepository
     * @param TransactionRepository $transactionRepository
     */
    public function __construct(
        ClaimRepository $claimRepository,
        OfferRepository $offerRepository,
        TransactionRepository $transactionRepository
    ) {
        $this->claimRepository          = $claimRepository;
        $this->offerRepository          = $offerRepository;
        $this->transactionRepository    = $transactionRepository;
    }

    /**
     * This function Tests a ClaimParticipant for exceeding a Promotion Cap limit
     *
     * @param \Ignite\Claim\Entities\ClaimParticipant|int $claimParticipant
     *
     * @return \stdClass
     */
    public function checkCap($claimParticipant)
    {
        $checkResponse = new \stdClass;
        $checkResponse->result = true;
        $checkResponse->data   = false;
        $checkResponse->errors = [];

        // ---------------------------------
        // Get Claim Participant and Supporting Tables from Database
        // ---------------------------------
        if (! is_array($claimParticipant) || ! isset($claimParticipant['lineitems']) || ! isset($claimParticipant['lineitems'][0]['offer'])) {
            // -----------------
            // Check if we got a ClaimParticipant, or just an Id
            // -----------------
            if (! is_array($claimParticipant)) {
                $claimParticipantId = $claimParticipant;
            } else {
                $claimParticipantId = $claimParticipant['id'];
            }

            $response = $this->claimRepository->get([
                'class' => 'claim_participant',
                'action' => 'find',
                'mode' => 'array',
                'with' => ['claim', 'lineitems', 'lineitems.offer'],
                'id' => $claimParticipantId
            ]);
            $claimParticipant = $response->data;
        }

        // ---------------------------------
        // Get Promotion for Claim and any Caps on the Promotion
        // ---------------------------------
        if (! is_array($claimParticipant) || ! isset($claimParticipant['lineitems']) || ! isset($claimParticipant['lineitems'][0]['offer'])) {
            return $checkResponse;
        }

        $offer = $claimParticipant['lineitems'][0]['offer'];
        $promotionId = $offer['promotion_id'];
        $response = $this->offerRepository->get([
            'class' => 'claim_offer_promotion',
            'action' => 'find',
            'mode' => 'array',
            'with' => ['caps'],
            'id' => $promotionId
        ]);
        $promotion = $response->data;

        // ---------------------------------
        // Check for Active Promotion Cap where Claim Date matches
        // ---------------------------------
        $claimDate      = $claimParticipant['claim']['created_at'];
        $claimDateSecs  = strtotime($claimDate);
        $claimMonth     = date('m', $claimDateSecs);
        $transactions   = false;

        foreach ($promotion['caps'] as $cap) {
            // -------------------
            // Check if Cap is Active
            // -------------------
            if (! $cap['status']) {
                continue;
            }

            // -------------------
            // Check Cap Start and End Date Range
            // -------------------
            $capStartDateSecs   = (! empty($cap['start_date'])) ? strtotime($cap['start_date']) : 0;
            $capEndDateSecs     = (! empty($cap['end_date']))   ? strtotime($cap['end_date'])   : 0;

            if ($capStartDateSecs && $claimDateSecs < $capStartDateSecs) {
                continue;
            }

            if ($capEndDateSecs && $claimDateSecs > $capEndDateSecs) {
                continue;
            }

            // -------------------
            // Check Cap Period Date Range
            // -------------------
            if ($cap['max_period'] != '') {
                $period = self::getPeriodForDate($claimDate, $cap['max_period'], $cap['fiscal_month'], 'U');

                if ($claimDateSecs < $period['startDate']) {
                    continue;
                }

                if ($claimDateSecs > $period['endDate']) {
                    continue;
                }

                $capStartDateSecs   = $period['startDate'];
                $capEndDateSecs     = $period['endDate'];
            }

            // --------------------------------------------------------
            // If we got here, we got a Date Range Match
            // So we need to check the Cap Max Value for the Transactions in that Range
            // --------------------------------------------------------
            if ($transactions === false) {
                $transactions = $this->transactionRepository->findByUser($claimParticipant['user_id']);
            }

            // -------------------
            // Get Total for Transactions in the Date Range
            // -------------------
            $totalEarned = 0;

            foreach ($transactions as $transaction) {
                // -------------------
                // Only consider Transaction type EARNED
                // -------------------
                if ($transaction['type'] != 'EARNED') {
                    continue;
                }

                // -------------------
                // Check if Transaction is within the Cap Date Range
                // -------------------
                $transactionDateSecs =  strtotime($transaction['created_at']);

                if ($capStartDateSecs && $transactionDateSecs  < $capStartDateSecs) {
                    continue;
                }

                if ($capEndDateSecs && $transactionDateSecs > $capEndDateSecs) {
                    continue;
                }

                $totalEarned += $transaction['value'];
            }

            // -------------------
            // Check if Claim value added to Transaction total would exceed Cap Max Value
            // -------------------
            $claimValue = $claimParticipant['value'];
            if ($claimValue + $totalEarned > $cap['max_value']) {
                $checkResponse->result   = false;
                $checkResponse->data     = $cap;
                $checkResponse->errors[] = sprintf(
                    "Claim would exceed Cap: \"%s\" Max: %s",
                    $cap['name'],
                    number_format($cap['max_value'])
                );
                $checkResponse->errors[] = sprintf(
                    "Claim Date: %s, Amount: %s Earned: %s Total: %s",
                    date('m/d/Y', $claimDateSecs),
                    number_format($claimValue),
                    number_format($totalEarned),
                    number_format($claimValue + $totalEarned)
                );

                $periodText = 'none';
                if (! empty($cap['max_period'])) {
                    $periodText = (! empty(OfferPromotionCap::$maxPeriods[$cap['max_period']]))
                        ? OfferPromotionCap::$maxPeriods[$cap['max_period']]
                        : sprintf("Unknown '%s'", $cap['max_period']);
                }

                $checkResponse->errors[] = sprintf(
                    "Period: %s, Start Date: %s, End Date: %s",
                    $periodText,
                    $capStartDateSecs ? date('m/d/Y', $capStartDateSecs) : 'none',
                    $capEndDateSecs ? date('m/d/Y', $capEndDateSecs) : 'none'
                );
                break;
            }
        }

        return $checkResponse;
    }

    /**
     * This function returns a StartDate, EndDate, and Period Number for a Date
     *
     * @param  string  $date
     * @param  string  $periodType
     * @param  integer $fiscalMonth
     * @param  string  $dateFormat
     * @return array
     */
    public static function getPeriodForDate($date, $periodType, $fiscalMonth = 1, $dateFormat = 'm/d/Y')
    {
        $dateSecs = strtotime($date);

        if (! $dateSecs) {
            return ['error' => sprintf("Invalid Date: %s", $date)];
        }

        $dateFormatted  = date('m-d-Y', $dateSecs);
        $dateParts      = explode('-', $dateFormatted);
        $month          = $dateParts[0];
        $day            = $dateParts[1];
        $year           = $dateParts[2];

        // ---------------------------------
        // Determine Fiscal Year Dates
        // ---------------------------------
        if ($month < $fiscalMonth) {
            $startFiscalSecs = strtotime($year - 1 . '-' . $fiscalMonth . '-' . '01');
        } else {
            $startFiscalSecs = strtotime($year . '-' . $fiscalMonth . '-' . '01');
        }

        switch ($periodType) {
            case 'year':
                $startDateSecs  = $startFiscalSecs;
                $endDateSecs    = strtotime('+1 year', $startFiscalSecs);
                $periodNumber   = 1;
                break;

            case 'semi':
                for ($period = 0; $period < 2; ++$period) {
                    $startDateSecs  = strtotime('+' . $period * 6 . ' months', $startFiscalSecs);
                    $endDateSecs    = strtotime('+' . ($period + 1) * 6 . ' months', $startFiscalSecs);
                    $periodNumber   = $period + 1;
                    if ($dateSecs >= $startDateSecs && $dateSecs < $endDateSecs) {
                        break;
                    }
                }
                break;

            case 'quarter':
                for ($period = 0; $period < 4; ++$period) {
                    $startDateSecs  = strtotime('+' . $period * 3 . ' months', $startFiscalSecs);
                    $endDateSecs    = strtotime('+' . ($period + 1) * 3 . ' months', $startFiscalSecs);
                    $periodNumber   = $period + 1;
                    if ($dateSecs >= $startDateSecs && $dateSecs < $endDateSecs) {
                        break;
                    }
                }
                break;

            case 'month':
                $startDateSecs = strtotime("$year-$month-01", $startFiscalSecs);
                $endDateSecs   = strtotime('+1 month', $startDateSecs);
                $periodNumber  = ($month < $fiscalMonth) ? $month + 12 - $fiscalMonth + 1 : $month - $fiscalMonth + 1 ;
                break;

            default:
                return ['error' => sprintf("Invalid Period Type: %s", $periodType)];
                break;
        }

        if (! $startDateSecs) {
            return ['error' => sprintf("Could not Calculate Period Start Date")];
        }

        if (! $endDateSecs) {
            return ['error' => sprintf("Could not Calculate Period End Date")];
        }

        $startDate = date($dateFormat, $startDateSecs);
        $endDate = date($dateFormat, $endDateSecs);

        return ['startDate' => $startDate, 'endDate' => $endDate, 'period' => $periodNumber];
    }
}
