<?php

namespace Ignite\Claim\Models\Dashboard\Charts;

use Ignite\Core\Contracts\Dashboard\WithPercentageChange;
use Ignite\Core\Models\Dashboard\Concerns\DeterminesPercentageChange;
use Ignite\Core\Models\Dashboard\Stat;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\DB;

abstract class ClaimMetric extends Stat implements WithPercentageChange
{
    use DeterminesPercentageChange;

    /**
     * Create a new stat instance.
     *
     * @param  array|object $attributes
     */
    public function __construct($attributes = [])
    {
        parent::__construct(array_merge([
            'type' => 'metric',
        ], $attributes));
    }

    /**
     * The chart size in the layout.
     *
     * @return string
     */
    public function getSize()
    {
        return $this->get('size', 'col-xs-6 col-sm-3');
    }

    /**
     * Get the correct data value from storage.
     *
     * @return mixed
     */
    public function getData()
    {
        if (! $this->get('data', false)) {
            $current = $this->scope($this->getValueQuery($this->getDateRange()))->count() ?? 0;
            $this->attributes['data'] = $current;

            if ($this instanceof WithPercentageChange) {
                $previous = $this->scope($this->getValueQuery($this->getPreviousDateRange()))->count() ?? 0;
                $this->setPercentageChange($current, $previous);
            }
        }

        return $this->get('data');
    }

    /**
     * @param Builder $query
     *
     * @return Builder
     */
    abstract protected function scope(Builder $query);

    /**
     * The query to get the scoped count of claims.
     *
     * @param array $range
     *
     * @return Builder
     */
    protected function getValueQuery($range)
    {
        return DB::query()
             ->select(DB::raw('COUNT(id) as total'))
             ->from('claim_participant')
             ->whereNull('deleted_at')
             ->whereBetween('created_at', $range);
    }

    /**
     * @inheritdoc
     */
    public function setPercentageChange($current, $previous)
    {
        $this->offsetSet('current', number_format($current, 0));
        $this->offsetSet('previous', number_format($previous, 0));
        $this->offsetSet('direction', $this->determineChangeType($current, $previous));
        $this->offsetSet('percentage', $this->determinePercentageChange($current, $previous));
    }
}
