<?php

namespace Ignite\Claim\Repositories;

use Illuminate\Support\Facades\DB;

class OfferRepository extends CoreRepository
{
    public static $offerFilters = [
        'promotion_id' => ['type' => 'integer'],
        'type'         => ['type' => 'string'],
        'offer_group'  => ['type' => 'string'],
        'name'         => ['type' => 'string'],
        'sku'          => ['type' => 'string'],
        'status'       => ['type' => 'integer'],
        'start_date'   => ['type' => 'date', 'operator' => '>='],
        'end_date'     => ['type' => 'date', 'operator' => '<='],
        'grace_date'   => ['type' => 'date', 'operator' => '<='],
    ];

    public static $offerPromotionCampaignFilters = [
        'status'                => ['type' => 'integer'],
    ];

    public static $offerPromotionFilters = [
        'campaign_id'           => ['type' => 'integer'],
        'promotion_group_id'    => ['type' => 'integer'],
        'status'                => ['type' => 'integer'],
        'start_date'            => ['type' => 'date',       'operator' => '>='],
        'end_date'              => ['type' => 'date',       'operator' => '<='],
    ];

    public static $offerPromotionGroupFilters = [
        'status'                => ['type' => 'integer'],
    ];

    public static $offerUserPromotionGroupFilters = [
        'user_id'               => ['type' => 'integer'],
        'promotion_group_id'    => ['type' => 'integer'],
    ];

    public static $offerValueFilters = [
        'offer_id'              => ['type' => 'integer'],
        'value_type_id'         => ['type' => 'integer'],
        'participant_type'      => ['type' => 'string'],
    ];

    public static $offerPromotionCapFilters = [
        'offer_promotion_id'    => ['type' => 'integer'],
        'name'                  => ['type' => 'string'],
        'start_date'            => ['type' => 'date'],
        'end_date'              => ['type' => 'date'],
        'fiscal_month'          => ['type' => 'integer'],
        'status'                => ['type' => 'integer'],
    ];

    public function __construct()
    {
        $classes = [
            [
                'path'              => \Ignite\Claim\Entities\Offer::class,
                'filters'           => self::$offerFilters,
                'defaultTypesKey'   => 'module.offer.types.*',
                'nickName'          => 'claim_offer'
            ],
            [
                'path'              => \Ignite\Claim\Entities\OfferCampaign::class,
                'filters'           => self::$offerPromotionCampaignFilters,
                'nickName'          => 'claim_offer_campaign',
            ],
            [
                'path'              => \Ignite\Claim\Entities\OfferPromotion::class,
                'filters'           => self::$offerPromotionFilters,
                'nickName'          => 'claim_offer_promotion',
            ],
            [
                'path'              => \Ignite\Claim\Entities\OfferPromotionGroup::class,
                'filters'           => self::$offerPromotionGroupFilters,
                'nickName'          => 'claim_offer_promotion_group',
            ],
            [
                'path'              => \Ignite\Claim\Entities\OfferUserPromotionGroup::class,
                'filters'           => self::$offerUserPromotionGroupFilters,
                'nickName'          => 'claim_offer_user_promotion_group',
            ],
            [
                'path'              => \Ignite\Claim\Entities\OfferValue::class,
                'filters'           => self::$offerValueFilters,
                'nickName'          => 'claim_offer_value',
            ],
            [
                'path'              => \Ignite\Claim\Entities\OfferPromotionCap::class,
                'filters'           => self::$offerPromotionCapFilters,
                'nickName'          => 'claim_offer_promotion_cap',
            ],
        ];

        parent::__construct($classes);
    }

    public function get($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['action', 'class'];
        if ( ! $this->checkRequireds($function, $params, $requireds) )
            return $this->response;

        $action = $params['action'];

        switch($action)
        {
            case 'getOfferGroups':
                $this->getOfferGroups($params);
                break;

            default:
                parent::get($params);
                break;
        }

        return $this->response;
    }

    private function getOfferGroups($requestParams)
    {
        $function = __FUNCTION__;

        $requireds = ['promotion_id'];
        if ( ! $this->checkRequireds($function, $requestParams, $requireds) )
            return false;

        $promotionId = $requestParams['promotion_id'];

        // ------------------------------
        // Get Offer Groups/Offers
        // ------------------------------
        $params = ['class' => 'claim_offer', 'action' => 'getList', 'promotion_id' => $promotionId];
        $params = array_merge($params, parent::getFilterParams('offer', $requestParams, $params));
        $response = parent::get($params);

        if ( ! $response->result ) {
            $message = sprintf("Unable to get Groups / Offers for Promotion Id = '%s'.", $promotionId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        // ------------------------------
        // Organize into Nested Array of [Offer Groups][Offers]
        // ------------------------------
        $offers = $response->data;

        $offerGroups = [];
        foreach($offers as $offer) {
            $names = [];
            $names['offer_id'] = $offer['id'];
            $names['name'] = $offer['name'];
            $offerGroups[$offer['offer_group']][] = $names;
        }

        $this->response->data = $offerGroups;

        return true;
    }

    /**
     * Put method.
     *
     * @param  array $params
     * @return bool|array
     */
    public function put($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['action', 'class'];
        if ( ! $this->checkRequireds($function, $params, $requireds) )
            return $this->response;

            $action = $params['action'];

            switch($action)
            {
                case 'createOffer':
                    $rc = $this->createOffer($params);
                    break;

                case 'deleteOffer':
                    $rc = $this->deleteOffer($params);
                    break;

                case 'updateOffer':
                    $rc = $this->updateOffer($params);
                    break;

                case 'createPromotion':
                    $rc = $this->createPromotion($params);
                    break;

                case 'deletePromotion':
                    $rc = $this->deletePromotion($params);
                    break;

                case 'updatePromotion':
                    $rc = $this->updatePromotion($params);
                    break;

                default:
                    $this->response->result     = false;
                    $this->response->errors[]   = sprintf("%s: Unknown action param '%s'.", $function, $action);
                    break;
            }

            return $this->response;
    }

    // ----------------------------------------------------------
    // ----------------------------------------------------------
    // ----------------------------------------------------------
    // Offer Methods
    // ----------------------------------------------------------
    // ----------------------------------------------------------
    // ----------------------------------------------------------
    private function createOffer($formData)
    {
        $function = __FUNCTION__;

        // -----------------------------------------
        // Create New Offer record
        // -----------------------------------------
        $offerData = $formData;

        $offerData['class'] = 'claim_offer';

        DB::beginTransaction();

        $response = parent::create($offerData);

        if ( ! $response->result ) {
            DB::rollBack();
            $message = sprintf("Unable to Create New Offer record.");
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $offer = $response->data;
        $offerId = $offer['id'];

        // ---------------------------------
        // Create Participant Value Records for Offer
        // ---------------------------------
        $class = 'claim_offer_value';
        $items = $formData['values'];

        if ( count($items) > 0 ) {
            foreach($items as $index => $tmpItem) {
                $item = &$items[$index];
                $item['class']      = $class;
                $item['offer_id']   = $offerId;
            }

            $response = parent::create($items);

            if ( ! $response->result ) {
                DB::rollBack();
                $message = sprintf("Problem Adding '%s' Child Records for Offer Id = '%s'\n", $class, $offerId);
                array_unshift($response->errors, $message);
                $this->response = $response;
                return false;
            }
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        return true;
    }

    private function deleteOffer($formData)
    {
        $function = __FUNCTION__;

        $requireds = ['offer_id'];
        if ( ! $this->checkRequireds($function, $formData, $requireds) )
            return false;

        // ---------------------------------
        // Get Offer Data from Database
        // ---------------------------------
        $offerId = $formData['offer_id'];

        $params = ['class' => 'claim_offer', 'action' => 'find', 'mode' => 'array',
            'with' => ['values'], 'id' => $offerId];

        $response = parent::get($params);

        if ( ! $response->result ) {
            $message = sprintf("Can't find Offer for Id = '%s'\n", $offerId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $offer = $response->data;

        // ---------------------------------
        // Delete Participant Value Records for Offer
        // ---------------------------------
        DB::beginTransaction();

        $class = 'claim_offer_value';
        $items = $offer['values'];

        if ( count($items) > 0 ) {
            foreach($items as $index => $tmpItem) {
                $item = &$items[$index];
                $item['class']     = $class;
            }

            $response = parent::delete($items);

            if ( ! $response->result ) {
                DB::rollBack();
                $message = sprintf("Problem Deleting '%s' Child records for Offer Id = '%s'\n", $class, $offerId);
                array_unshift($response->errors, $message);
                $this->response = $response;
                return false;
            }
        }

        // ---------------------------------
        // Delete Offer
        // ---------------------------------
        $offer['class'] = 'claim_offer';

        $response = parent::delete($offer);

        if ( ! $response->result ) {
            DB::rollBack();
            $message = sprintf("Could not Delete Offer record for Offer Id = '%s'\n", $offerId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        return true;
    }

    private function updateOffer($formData)
    {
        $function = __FUNCTION__;

        $offerData = $formData;

        // -----------------------------------------
        // Update Offer record
        // -----------------------------------------
        $offerData['class'] = 'claim_offer';
        $offerId = $offerData['id'];

        DB::beginTransaction();

        $response = parent::update($offerData);

        if ( ! $response->result ) {
            DB::rollBack();
            $message = sprintf("Unable to Update Offer record.");
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $offer = $response->data;

        // ---------------------------------
        // Update Participant Value Records for Offer
        // ---------------------------------
        $class = 'claim_offer_value';
        $items = $formData['values'];

        $foreignKeyName     = 'offer_id';
        $foreignKeyValue    = $offerId;

        if ( count($items) > 0 ) {
            foreach($items as $index => $tmpItem) {
                $item = &$items[$index];
                $item['class']      = ( empty($item['class']) )     ? $class    : $item['class'];
                $item['offer_id']   = ( empty($item['offer_id']) )  ? $offerId  : $item['offer_id'];
            }
        }

        $response = parent::sync($items, $class, $foreignKeyName, $foreignKeyValue);

        if ( ! $response->result ) {
            DB::rollBack();
            $message = sprintf("Problem Updating '%s' Child records for Offer Id = '%s'\n", $class, $offerId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        return true;
    }

    // ----------------------------------------------------------
    // ----------------------------------------------------------
    // ----------------------------------------------------------
    // Promotion Methods
    // ----------------------------------------------------------
    // ----------------------------------------------------------
    // ----------------------------------------------------------
    private function createPromotion($formData)
    {
        $function = __FUNCTION__;

        // -----------------------------------------
        // Create New Promotion record
        // -----------------------------------------
        $promotionData = $formData;

        $promotionData['class'] = 'claim_offer_promotion';

        DB::beginTransaction();

        $response = parent::create($promotionData);

        if ( ! $response->result ) {
            DB::rollBack();
            $message = sprintf("Unable to Create New Promotion record.");
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $promotion = $response->data;
        $promotionId = $promotion['id'];

        // ---------------------------------
        // Create Promotion Cap Records for Promotion
        // ---------------------------------
        $class = 'claim_offer_promotion_cap';
        $items = $formData['caps'];

        if ( count($items) > 0 ) {
            foreach($items as $index => $tmpItem) {
                $item = &$items[$index];
                $item['class']                  = $class;
                $item['offer_promotion_id']     = $promotionId;
            }

            $response = parent::create($items);

            if ( ! $response->result ) {
                DB::rollBack();
                $message = sprintf("Problem Adding '%s' Child Records for Promotion Id = '%s'\n", $class, $promotionId);
                array_unshift($response->errors, $message);
                $this->response = $response;
                return false;
            }
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        return true;
    }

    private function updatePromotion($formData)
    {
        $function = __FUNCTION__;

        $promotionData = $formData;

        // -----------------------------------------
        // Update Promotion record
        // -----------------------------------------
        $promotionData['class'] = 'claim_offer_promotion';
        $promotionId = $promotionData['id'];

        DB::beginTransaction();

        $response = parent::update($promotionData);

        if ( ! $response->result ) {
            DB::rollBack();
            $message = sprintf("Unable to Update Promotion record.");
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $promotion = $response->data;

        // ---------------------------------
        // Update Promotion Cap Records for Promotion
        // ---------------------------------
        $class = 'claim_offer_promotion_cap';
        $items = $formData['caps'];

        $foreignKeyName     = 'offer_promotion_id';
        $foreignKeyValue    = $promotionId;

        if ( count($items) > 0 ) {
            foreach($items as $index => $tmpItem) {
                $item = &$items[$index];
                $item['class']                  = ( empty($item['class']) )     ? $class    : $item['class'];
                $item['offer_promotion_id']     = ( empty($item['offer_promotion_id']) )  ? $promotionId  : $item['offer_promotion_id'];
            }
        }

        $response = parent::sync($items, $class, $foreignKeyName, $foreignKeyValue);

        if ( ! $response->result ) {
            DB::rollBack();
            $message = sprintf("Problem Updating '%s' Child records for Promotion Id = '%s'\n", $class, $promotionId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        return true;
    }

    private function deletePromotion($formData)
    {
        $function = __FUNCTION__;

        $requireds = ['promotion_id'];
        if ( ! $this->checkRequireds($function, $formData, $requireds) )
            return false;

        // ---------------------------------
        // Get Promotion Data from Database
        // ---------------------------------
        $promotionId = $formData['promotion_id'];

        $params = ['class' => 'offer_promotion', 'action' => 'find', 'mode' => 'array',
            'with' => ['caps'], 'id' => $promotionId];

        $response = parent::get($params);

        if ( ! $response->result ) {
            $message = sprintf("Can't find Promotion for Id = '%s'\n", $promotionId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $promotion = $response->data;

        // ---------------------------------
        // Delete Promotion Cap Records for Promotion
        // ---------------------------------
        DB::beginTransaction();

        $class = 'claim_offer_promotion_cap';
        $items = $promotion['caps'];

        if ( count($items) > 0 ) {
            foreach($items as $index => $tmpItem) {
                $item = &$items[$index];
                $item['class']     = $class;
            }

            $response = parent::delete($items);

            if ( ! $response->result ) {
                DB::rollBack();
                $message = sprintf("Problem Deleting '%s' Child records for Promotion Id = '%s'\n", $class, $promotionId);
                array_unshift($response->errors, $message);
                $this->response = $response;
                return false;
            }
        }

        // ---------------------------------
        // Delete Promotion
        // ---------------------------------
        $promotion['class'] = 'offer_promotion';

        $response = parent::delete($promotion);

        if ( ! $response->result ) {
            DB::rollBack();
            $message = sprintf("Could not Delete Promotion record for Promotion Id = '%s'\n", $promotionId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        // ---------------------------------
        // Commit DB Transaction and return success
        // ---------------------------------
        DB::commit();

        return true;
    }

}
