<?php

namespace Ignite\Claim\Http\Controllers\Admin;

use Ignite\Claim\Entities\Table;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Claim\Entities\Dynamic\IgniteDB;
use Ignite\Claim\Entities\Dynamic\DynamicModel;

class TableColumnsController extends Controller
{
    public $formFields = [
        'name' => [
            'formName'   => 'Column Name',
            'dbType'     => 'string',
            'required'   => true,
            'formType'   => 'text',
            'default'    => 'new',
            'editStatic' => false,
        ],
        'displayName' => [
            'formName'   => 'Display Name',
            'dbType'     => 'string',
            'required'   => false,
            'formType'   => 'text',
            'default'    => '',
            'editStatic' => true,
        ],
        'reportName' => [
            'formName'   => 'Report Name',
            'dbType'     => 'string',
            'required'   => false,
            'formType'   => 'text',
            'default'    => '',
            'editStatic' => true,
        ],
        'type' => [
            'formName'   => 'Type',
            'dbType'     => 'string',
            'required'   => true,
            'formType'   => 'select',
            'default'    => '',
            'options'    => [],
            'editStatic' => false,
        ],
        'null' => [
            'formName'   => 'Nullable',
            'dbType'     => 'bool',
            'required'   => false,
            'formType'   => 'checkbox',
            'default'    => true,
            'editStatic' => false,
        ],
        'required' => [
            'formName'   => 'Required',
            'dbType'     => 'bool',
            'required'   => false,
            'formType'   => 'checkbox',
            'default'    => true,
            'editStatic' => true,
        ],
        'sensitive' => [
            'formName'   => 'Sensitive',
            'dbType'     => 'bool',
            'required'   => false,
            'formType'   => 'checkbox',
            'default'    => true,
            'editStatic' => true,
        ],
        'format' => [
            'formName'   => 'Format',
            'dbType'     => 'string',
            'required'   => false,
            'formType'   => 'text',
            'default'    => '',
            'editStatic' => true,
        ],
        'validateType' => [
            'formName'   => 'Validation Type',
            'dbType'     => 'string',
            'required'   => false,
            'formType'   => 'select',
            'default'    => '',
            'editStatic' => true,
        ],
        'validate' => [
            'formName'   => 'Laravel Validation String',
            'dbType'     => 'string',
            'required'   => false,
            'formType'   => 'text',
            'default'    => '',
            'editStatic' => true,
        ],
        'options' => [
            'formName'   => 'Select Options',
            'dbType'     => 'none',
            'required'   => false,
            'formType'   => 'textarea',
            'default'    => '',
            'editStatic' => true,
        ],
    ];

    public function __construct()
    {
        $this->init();
        $this->middleware(['auth']);
    }

    private function init()
    {
        $this->formFields['type']['options'] = DynamicModel::$dbColumnTypes;
    }

    public function index()
    {
        $tables = Table::all();

        return view('Claim::admin.settings.tables.index', compact('tables'));
    }

    public function create()
    {
        $haveTables = Table::all()->toArray();
        $tables = array_keys(DynamicModel::$tables);
        $tablesLookup = array_flip($tables);

        foreach($haveTables as $index => $haveTable) {
            $keyParts = explode('.', $haveTable['key']);
            $tableName = end($keyParts);
            if (isset($tablesLookup[$tableName])) {
                unset($tables[$tablesLookup[$tableName]]);
            } else {
                $tables[$tableName] = $tableName;
            }
        }

        return view('Claim::admin.settings.tables.create', compact('tables'));
    }

    public function edit($tableColumnsKey = '')
    {
        $errors = [];

        // ----------------------------------
        // Get Table portion of Key
        // ----------------------------------
        $parts = explode('.', $tableColumnsKey);
        $tableName = end($parts);

        // ----------------------------------
        // Check if we have a known Dynamic Table
        // ----------------------------------
        if ($tableName == '') {
            flash()->error("You must select a table first.")->important();
            return back();
        }

        if (empty(DynamicModel::$tables[$tableName]) || ! isset(DynamicModel::$tables[$tableName]['class'])) {
            flash()->error(sprintf(
                'WARNING: Unknown table "%s", please update the \Ignite\Core\Entities\Dynamic\DynamicModel::$tables property.',
                $tableName
            ))->important();
            return back();
        }

		$isDynamic = true;
        $class = DynamicModel::$tables[$tableName]['class'];
        if ( ! is_subclass_of($class, DynamicModel::class)) {
			$isDynamic = false;
            $errors[] = "WARNING: The '$tableName' Table is not Derived from DynamicModel, and cannot be used for Dynamic Columns.";
        }

        $autoSetFields = [];
		if ($isDynamic) {
			$table = new $class($autoSetFields);
			if (! $table->hasDynamic()) {
				$dbTableName        = $table->getTable();
				$dynamicColumnName  = $table->getDynamicColumnName();
				$errors[] = "WARNING: The '$dbTableName' Table is missing the Dynamic Column '$dynamicColumnName' on the Database.";
				$errors[] = "You can configure the Dynamic Columns, but they won't actually work.";
			}
		}
		else {
			$table = new $class();
		}

        // ----------------------------------
        // Get Table Configuration from Settings
        // ----------------------------------
        $key = DynamicModel::tableSettingKey($tableName);
        $tableModel = Table::findByKey($key);

        if ($tableModel) {
            $dbColumnsChanged = $this->checkDbColumnsChanged($table, $errors);
        } else {
            // Default to Static Columns from Database
            $class = DynamicModel::$tables[$tableName]['class'];
            $dbTableName = $table->getTable();
            $tmpDbColumns = IgniteDB::getColumns($dbTableName);
            $dbColumns = new \stdClass();
            foreach($tmpDbColumns as $columnName => $tmpDbColumn) {
                $dbColumn = new \stdClass();
                foreach($tmpDbColumn as $fieldName => $value) {
                    $dbColumn->$fieldName = $value;
                }
                $dbColumns->$columnName = $dbColumn;
            }

            $tableModel = new Table();
            $tableModel->key = $key;
            $tableModel->name = $dbTableName;
            $tableModel->fqn = $class;
            $tableModel->static_columns = (array) $dbColumns;
            $tableModel->dynamic_columns = [];

            $errors[] = sprintf("WARNING: There is no entry created for the '%s' table with key '%s'. Pressing Save on this page will create a new table entry.", $tableName, $key);

            $dbColumnsChanged = false;
        }

		// ----------------------------------
		// If we have a Columns Consistancy issue, add User Information
		// ----------------------------------
		$infos = [];
		if ($dbColumnsChanged) {
			$infos[] = "To fix the database columns 'Out-of-Sync' problem:";
			$infos[] = "Copy the columns down from the error message above.";
			$infos[] = "Press the 'Sync DB Columns' button below.";
			$infos[] = "After the page refreshes, update the columns you took notes on.";
			$infos[] = "NOTE: If the table is NOT dynamic, you have to fix the columns on the class.";
			$infos[] = "BE SURE TO UPDATE ANY CODE THAT IS DEPENDENT ON THESE COLUMNS.";
		}

        $pageData = [
            'isDynamic'         => $isDynamic,
            'infos'             => $infos,
            'dbColumnsChanged'  => $dbColumnsChanged,
            'tableModel'        => $tableModel,
            'columnTypes'       => DynamicModel::$dbColumnTypes,
            'validateTypes'     => DynamicModel::getValidateTypes(),
            'formFields'        => $this->formFields,
        ];

        return view('Claim::admin.settings.tables.edit', $pageData)->with('errors', $errors);
    }

    public function delete($key)
    {
        $table = Table::findByKey($key);

        if (empty($table)) {
            flash()->warning(sprintf("No table exists with key: '%s'.", $key))->important();
            return back();
        }

        try {
            $table->delete();
        } catch (\Exception $e) {
            flash()->error(sprintf("Unable to delete table configuration for key: '%s'.", $key))->important();
            return back();
        }

        flash()->success(sprintf("Successfully deleted table configuration with key: '%s'.", $key))->important();
        return back();
    }

    private function checkDbColumnsChanged($table, &$errors)
    {
        $tmpErrors = [];

        if (! empty($table->isDynamicModel)) {
            $classOrDatabase = 'Database';
            $dbColumns = $table->dynamic->getDbColumns();
        } else {
            $classOrDatabase = 'Class';
            $dbColumns = $table::getFields();
        }

        $nativeColumns = IgniteDB::getColumnsNative($table->getTable());

        foreach($dbColumns as $name => $column) {
            if (! isset($nativeColumns[$name])) {
                $tmpErrors[] = sprintf("Settings Table '%s' : Column '%s' : missing from %s Columns.", $table->getTable(), $name, $classOrDatabase);
            }
        }

        foreach($nativeColumns as $name => $column) {
            // Skip Eloquent "Built-In" columns
            if (in_array($name, ['created_at', 'created_by', 'updated_at', 'updated_by', 'deleted_at', 'deleted_by'])) {
                continue;
            }

            if (! isset($dbColumns[$name])) {
                $tmpErrors[] = sprintf("The column '%s' is missing from the table columns configuration.", $name);
            }
        }

        if (! empty($tmpErrors)) {
            $message = sprintf("The actual database table for '%s' and the table columns configuration are out of sync:", $table->getTable());
            array_unshift($tmpErrors, $message);
            $errors = array_merge($errors, $tmpErrors);
            return true;
        }

        return false;
    }
}
