<?php

namespace Ignite\Claim\Http\Controllers\Admin;

use Ignite\Flash\Facades\Flash;
use Illuminate\Http\Request;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Claim\Entities\ClaimParticipant;
use Ignite\Claim\Models\Status;
use Ignite\Claim\Models\Grid\ProcessTable;

class ProcessController extends Controller
{
    /**
     * Display a list of pending claims in need of processing.
     *
     * @param ProcessTable $table
     * @param Request      $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function index(ProcessTable $table, Request $request)
    {
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Claim::admin.process.index');
    }

    /**
     * Update the status for the provided claim ids.
     *
     * @param Request $request
     * @param Status  $helper
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Status $helper)
    {
        $data = $request->validate([
            'items' => 'required',
            'status' => 'required', // TODO: Should be approved, cancelled or declined
            'reason' => 'required_if:status,declined'
        ]);

        $ids = explode(',', $data['items']);
        $passed = $failed = 0;

        $claims = ClaimParticipant::whereIn('id', $ids)->where('status', 'pending')->get()->keyBy('id');

        // Loop through each claim participant and send them an email notification.
        foreach ($ids as $id) {
            if (! $claims->has($id)) {
                continue;
            }

            $claim = $claims->get($id);

            try {
                $response = $helper->setStatus($claim->getKey(), $data);
                if ($response->result) {
                    ++$passed;
                } else {
                    Flash::error(implode(',', $response->errors));
                    ++$failed;
                }
            } catch (\Exception $e) {
                $message = sprintf(
                    'Error: %s in %s on line %s',
                    $e->getMessage(),
                    $e->getFile(),
                    $e->getLine()
                );
                Flash::error($message);
                $this->logException($e);
                ++$failed;
            }
        }

        $helper->flashMessage(count($ids), $passed, $failed, $data['status']);

        return redirect(route('admin.claim.process'));
    }
}
