<?php

namespace Ignite\Claim\Http\Controllers\Admin;

use Ignite\Claim\Entities\ClaimParticipant;
use Ignite\Claim\Models\Grid\TrashedTable;
use Ignite\Flash\Facades\Flash;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Ignite\Claim\Services\Shredder;

class TrashedController extends Controller
{
    /**
     * Show all thrashed claim participant records.
     *
     * @param TrashedTable $table
     * @param Request $request
     *
     * @return JsonResponse|View
     */
    public function index(TrashedTable $table, Request $request)
    {
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Claim::admin.trashed');
    }

    /**
     * Restore the trashed claim participant.
     *
     * @param Request $request
     *
     * @return RedirectResponse
     */
    public function store(Request $request)
    {
        app('db')->beginTransaction();
        try {
            $ids = explode(',', $request->get('ids', ''));
            $claimParticipants = ClaimParticipant::query()
                                                 ->with('claim')
                                                 ->onlyTrashed()
                                                 ->whereIn('id', $ids)
                                                 ->get();

            foreach ($claimParticipants as $claimParticipant) {
                $claimParticipant->claim()->withTrashed()->restore();
                $claimParticipant->restore();
            }

            app('db')->commit();
            $message = sprintf(
                '%s trashed %s successfully restored.',
                count($ids),
                str_plural('claim participant', count($ids))
            );
            Flash::success($message);
        } catch (\Exception $e) {
            app('db')->rollback();
            Flash::error($e->getMessage());
        }

        return redirect()->route('admin.claim.trashed');
    }

    /**
     * Destroy the trashed claim participant.
     *
     * @param Request $request
     * @param Shredder $shredder
     *
     * @return RedirectResponse
     */
    public function destroy(Request $request, Shredder $shredder)
    {
        app('db')->beginTransaction();
        try {
            $ids = explode(',', $request->get('ids', ''));
            $count = $shredder->destroy($ids);
            $message = sprintf(
                '%s trashed %s successfully deleted.',
                $count,
                str_plural('claim participant', $count)
            );
            app('db')->commit();
            Flash::success($message);
        } catch (\Exception $e) {
            app('db')->rollback();
            Flash::error($e->getMessage());
        }

        return redirect()->route('admin.claim.trashed');
    }
}
