<?php

namespace Ignite\Claim\Repositories;

use Illuminate\Support\Facades\DB;
use Ignite\Claim\Entities\RuleCalculate;

class RuleRepository extends CoreRepository
{
    /**
     * The child class declarations.
     * @var array
     */
    private static $childClasses = [
        'claim_rule_condition'   => 'conditions',
        'claim_rule_value'       => 'values',
        'claim_rule_participant' => 'participants',
        'claim_rule_offer'       => 'offers',
    ];

    /**
     * The rule filter declarations.
     * @var array
     */
    public static $ruleFilters = [
        'offer_promotion_id' => ['type' => 'integer'],
        'name'               => ['type' => 'string'],
        'start_date'         => ['type' => 'date'],
        'end_date'           => ['type' => 'date'],
        'status'             => ['type' => 'integer'],
    ];

    /**
     * The rule condition filter declarations.
     * @var array
     */
    public static $ruleConditionFilters = [
        'rule_id' => ['type' => 'integer'],
    ];

    /**
     * The rule offer filter declarations.
     * @var array
     */
    public static $ruleOfferFilters = [
        'rule_id' => ['type' => 'integer'],
    ];

    /**
     * The rule participant filter declarations.
     * @var array
     */
    public static $ruleParticipantFilters = [
        'rule_id' => ['type' => 'integer'],
    ];

    /**
     * The rule value filter declarations.
     * @var array
     */
    public static $ruleValueFilters = [
        'rule_id' => ['type' => 'integer'],
    ];

    /**
     * The rule log filter declarations.
     * @var array
     */
    public static $ruleLogFilters = [
        'claim_participant_id' => ['type' => 'integer'],
    ];

    /**
     * RuleServiceProvider constructor.
     */
    public function __construct()
    {
        $classes = [
            [
                'path'    => \Ignite\Claim\Entities\Rule::class,
                'filters' => self::$ruleFilters,
                'nickName' => 'claim_rule',
            ],
            [
                'path'    => \Ignite\Claim\Entities\RuleCondition::class,
                'filters' => self::$ruleConditionFilters,
                'nickName' => 'claim_rule_condition',
            ],
            [
                'path'    => \Ignite\Claim\Entities\RuleOffer::class,
                'filters' => self::$ruleOfferFilters,
                'nickName' => 'claim_rule_offer',
            ],
            [
                'path'    => \Ignite\Claim\Entities\RuleParticipant::class,
                'filters' => self::$ruleParticipantFilters,
                'nickName' => 'claim_rule_participant',
            ],
            [
                'path'    => \Ignite\Claim\Entities\RuleValue::class,
                'filters' => self::$ruleValueFilters,
                'nickName' => 'claim_rule_value',
            ],
            [
                'path'    => \Ignite\Claim\Entities\RuleLog::class,
                'filters' => self::$ruleLogFilters,
                'nickName' => 'claim_rule_log',
            ],
        ];

        parent::__construct($classes);
    }

    /**
     * @todo: Explain the functionality.
     *
     * @param $params
     * @return bool
     */
    public function get($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['action', 'class'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $action = $params['action'];
        
        switch ($action) {
            case 'getClaimValue':
                $this->response = $this->getClaimValue($params);
                break;

            default:
                parent::get($params);
                break;
        }

        return $this->response;
    }

    /**
     * @todo: Explain the functionality.
     *
     * @param $params
     * @return mixed
     */
    public function put($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['action', 'class'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $action = $params['action'];
        
        switch ($action) {
            case 'createRule':
                $rc = $this->createRule($params);
                break;

            case 'deleteRule':
                $rc = $this->deleteRule($params);
                break;

            case 'updateRule':
                $rc = $this->updateRule($params);
                break;

            default:
                $this->response->result     = false;
                $this->response->errors[]   = sprintf("%s: Unknown action param '%s'.", $function, $action);
                break;
        }

        return $this->response;
    }

    /**
     * @todo: Explain the functionality.
     *
     * @param $formData
     * @return bool
     */
    private function createRule($formData)
    {
        $function = __FUNCTION__;

        // -----------------------------------------
        // Create New Rule record
        // -----------------------------------------
        $ruleData = $formData;

        $ruleData['class'] = 'claim_rule';

        DB::beginTransaction();

        $response = parent::create($ruleData);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Unable to Create New Rule record.");
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $rule = $response->data;
        $ruleId = $rule['id'];

        // ---------------------------------
        // Create Child Records for Rule
        // ---------------------------------
        foreach (self::$childClasses as $class => $itemsName) {
            $items = $formData[$itemsName];
            if (count($items) > 0) {
                foreach ($items as $index => $tmpItem) {
                    $item = &$items[$index];
                    $item['class']     = $class;
                    $item['rule_id']   = $ruleId;
                }

                $response = parent::create($items);

                if (! $response->result) {
                    DB::rollBack();
                    $message = sprintf("Problem Adding '%s' Child Records for Rule Id = '%s'\n", $class, $ruleId);
                    array_unshift($response->errors, $message);
                    $this->response = $response;
                    return false;
                }
            }
        }

        // Commit DB Transaction and return success
        DB::commit();

        return true;
    }

    /**
     * @todo: Explain the functionality.
     *
     * @param  array $formData
     * @return bool
     */
    private function deleteRule($formData)
    {
        if (! $this->checkRequireds(__FUNCTION__, $formData, ['rule_id'])) {
            return false;
        }

        // Get Rule Data from Database
        $ruleId = $formData['rule_id'];
        $params = [
            'class' => 'claim_rule',
            'action' => 'find',
            'mode' => 'array',
            'with' => ['participants', 'offers', 'values', 'conditions'],
            'id' => $ruleId
        ];
        $response = parent::get($params);

        // Error
        if (! $response->result) {
            $message = sprintf("Can't find Rule for Id = '%s'\n", $ruleId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $rule = $response->data;

        // Delete Child Records for Rule
        DB::beginTransaction();

        foreach (self::$childClasses as $class => $itemsName) {
            $items = $rule[$itemsName];
            if (count($items) > 0) {
                foreach ($items as $index => $tmpItem) {
                    $item = &$items[$index];
                    $item['class']     = $class;
                }

                $response = parent::delete($items);

                if (! $response->result) {
                    DB::rollBack();
                    $message = sprintf("Problem Deleting '%s' Child records for Rule Id = '%s'\n", $class, $ruleId);
                    array_unshift($response->errors, $message);
                    $this->response = $response;
                    return false;
                }
            }
        }

        // Delete Rule
        $rule['class'] = 'claim_rule';

        $response = parent::delete($rule);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Could not Delete Rule record for Rule Id = '%s'\n", $ruleId);
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        // Commit DB Transaction and return success
        DB::commit();

        return true;
    }

    /**
     * @todo: Explain the functionality.
     *
     * @param  array $formData
     * @return bool
     */
    private function updateRule($formData)
    {
        $function = __FUNCTION__;

        $ruleData = $formData;

        // Update Rule record
        $ruleData['class'] = 'claim_rule';
        $ruleId = $ruleData['id'];

        DB::beginTransaction();

        $response = parent::update($ruleData);

        if (! $response->result) {
            DB::rollBack();
            $message = sprintf("Unable to Update Rule record.");
            array_unshift($response->errors, $message);
            $this->response = $response;
            return false;
        }

        $rule = $response->data;

        // Create Child Records for Rule
        $foreignKeyName  = 'rule_id';
        $foreignKeyValue = $ruleId;

        foreach (self::$childClasses as $class => $itemsName) {
            $items = $formData[$itemsName];
            if (count($items) > 0) {
                foreach ($items as $index => $tmpItem) {
                    $item = &$items[$index];
                    $item['class']   = (empty($item['class'])) ? $class : $item['class'];
                    $item['rule_id'] = (empty($item['rule_id'])) ? $ruleId : $item['rule_id'];
                }
            }

            $response = parent::sync($items, $class, $foreignKeyName, $foreignKeyValue);

            if (! $response->result) {
                DB::rollBack();
                $message = sprintf("Problem Updating '%s' Child records for Rule Id = '%s'\n", $class, $ruleId);
                array_unshift($response->errors, $message);
                $this->response = $response;
                return false;
            }
        }

        // Commit DB Transaction and return success
        DB::commit();

        return true;
    }

    /**
     * Calculate the claim value from the rules.
     *
     * @param  array $params
     * @return \stdClass
     */
    private function getClaimValue($params)
    {
        return app(RuleCalculate::class, ['debug' => null])->value($params);
    }
}
