<?php

namespace Ignite\Claim\Http\Controllers\Admin;

use Exception;
use Ignite\Flash\Facades\Flash;
use Illuminate\Container\EntryNotFoundException;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Ignite\Claim\Entities\Form as FormEntity;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Entities\Audit;
use Ignite\Core\Entities\Note;
use Ignite\Claim\Models\Form;
use Ignite\Claim\Entities\Claim;
use Ignite\Claim\Models\ClaimUtil;
use Ignite\Claim\Models\Grid\ClaimTable;
use Ignite\Claim\Repositories\ClaimRepository;
use Ignite\Claim\Models\CapCheck;
use Ignite\Claim\Repositories\OfferRepository;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Contracts\Repositories\ParticipantRepository;
use Ignite\Core\Contracts\Repositories\TransactionRepository;

class ClaimController extends Controller
{
    /** @var ClaimRepository */
    private $claimRepository;

    /** @var ParticipantRepository */
    private $participantRepository;

    /** @var ClaimUtil */
    private $claimUtil;

    /** @var string */
    private static $claimFieldsExternalKey = 'form.claim.partial.claim_fields_external';

    /** @var string */
    private static $claimFieldsInternalKey = 'form.claim.partial.claim_fields_internal';

    /**
     * Create a new controller instance.
     *
     * @param ClaimRepository $claimRepository
     * @param ParticipantRepository $participantRepository
     */
    public function __construct(ClaimRepository $claimRepository, ParticipantRepository $participantRepository)
    {
        $this->middleware(['auth']);

        $this->claimRepository = $claimRepository;
        $this->participantRepository = $participantRepository;
        $this->claimUtil = new ClaimUtil($this->participantRepository);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return redirect('/admin/claim/list/all');
    }

    /**
     * Show the Claim List pages.
     *
     * @param  string     $status
     * @param  bool|int   $userId
     * @param  ClaimTable $table
     * @param  Request    $request
     * @return \Illuminate\Http\JsonResponse|RedirectResponse
     */
    public function list($status = '', $userId = false, ClaimTable $table, Request $request)
    {
        if ($status == '') {
            return redirect('/admin/claim/list/all');
        }

        $status = strtolower($status);
        session()->put('claimListType', $status);
        $table->with('status', $status);
        $user = null;

        if ($userId) {
            $table->with('user_id', $userId);
            $user = Participant::findOrFail($userId)->toArray();
        }

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Claim::admin.index', compact(
            'status',
            'userId',
            'user'
        ));
    }

    /**
     * Show the New Claim entry page.
     *
     * @param string $promotionId
     * @param string $code
     *
     * @return RedirectResponse
     * @throws Exception
     */
    public function create($promotionId = '', $code = '')
    {
        $errors = [];

        $returnURI = '/admin/claim/promotions';

        // ---------------------------------
        // Get Claim Submitter Info
        // ---------------------------------
        $submitter = $this->claimUtil->getSubmitterParticipant($errors);

        if (! $submitter) {
            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Check for no Promotion Selected
        // ------------------------------
        $offerSvc = app(OfferRepository::class);

        if ($promotionId == '') {
            $params   = [
                'class' => 'claim_offer_promotion',
                'action' => 'getList',
                'mode' => 'array',
                'status' => 1
            ];
            $response = $offerSvc->get($params);

            if (! $response->result) {
                $errors[] = "No active promotions.";
                $errors   = array_merge($errors, $response->errors);

                return redirect($returnURI)->with('errors', $errors);
            }

            $promotions = $response->data;

            // ------------------------------
            // If we have a Promotion count != 1, we need to go back to the Promotions list
            // ------------------------------
            if (count($promotions) != 1) {
                if (count($promotions) == 0) {
                    $message = 'No active promotions.';
                } else {
                    $message = 'Please select a promotion to make a claim.';
                }

                Flash::info($message);

                return redirect($returnURI);
            }

            // ------------------------------
            // If only 1 Promotion is Active we can stay here
            // and display the New Claim page for that Promotion
            // ------------------------------
            $promotionId = $promotions[0]['id'];
        }

        // ------------------------------
        // Check for New or Redisplay with Errors
        // ------------------------------
        $promotionHtml = '';
        $className     = 'claim';

        if ($code != '') {
            $formData = unserialize(session()->get($code));
            if (! $formData) {
                return $this->claimUtil->reEnterClaim($returnURI);
            }
        } else {
            $code                           = uniqid('');
            $formData                       = $this->claimRepository->toArray($className);
            $formData['claim_id']           = 'new';
            $formData['offer_promotion_id'] = $promotionId;
            $formData['claim_items']        = [];
            $formData['claim_participants'] = [];
            $formData['maxLineitems']       = setting('claim.max_lineitems', -1);
            $formData['maxParticipants']    = setting('claim.max_participants', 0);
        }

        $filesCode = $code;

        // ------------------------------
        // Get Form Fields
        // ------------------------------
        $claimFormSettings = FormEntity::findByKey(self::$claimFieldsExternalKey);

        if (! $claimFormSettings) {
            $errors[] = sprintf("Missing Form settings for Key = '%s'\n", self::$claimFieldsExternalKey);
            return redirect($returnURI)->with('errors', $errors);
        }

        $internalClaimFormSettings = FormEntity::findByKey(self::$claimFieldsInternalKey);

        if (! $internalClaimFormSettings) {
            $errors[] = sprintf("Missing Internal Form settings for Key = '%s'\n", self::$claimFieldsInternalKey);

            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Populate the Form Fields with the Form Data
        // ------------------------------
        $review      = false;
        $claimFields = Form::mergeFormFieldData(
            ['claim'],
            $formData,
            $claimFormSettings,
            $review,
            $errors
        );

        if (! $claimFields) {
            if (empty($claimFields)) {
                $message = sprintf("The list of external dynamic claim fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the external dynamic claim field data.");
            }
            array_unshift($errors, $message);

            return redirect($returnURI)->with('errors', $errors);
        }

        $internalClaimFields = Form::mergeFormFieldData(
            'claim_participant',
            $formData,
            $internalClaimFormSettings,
            $review,
            $errors
        );

        if (! $internalClaimFields) {
            if (empty($internalClaimFields)) {
                $message = sprintf("The list of internal dynamic claim fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the internal dynamic claim field data.");
            }
            array_unshift($errors, $message);

            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Get any Uploaded Files from previous save
        // ------------------------------
        $uploadFiles = (! empty($formData['upload_files'])) ? $formData['upload_files'] : [];

        // ------------------------------
        // Get Promotion
        // ------------------------------
        $params   = [
            'class' => 'claim_offer_promotion',
            'action' => 'find',
            'mode' => 'array',
            'id' => $promotionId
        ];
        $response = $offerSvc->get($params);

        if (! $response->result) {
            $errors[] = sprintf("Can't find an active promotion with ID: '%s'\n", $promotionId);
            $errors   = array_merge($errors, $response->errors);

            return redirect($returnURI)->with('errors', $errors);
        }

        $promotion = $response->data;

        // ------------------------------
        // Get Offer Groups/Offers
        // ------------------------------
        $offerSvc      = app(OfferRepository::class);
        $params        = [
            'class' => 'claim_offer',
            'action' => 'getOfferGroups',
            'mode' => 'array',
            'status' => 1,
            'promotion_id' => $promotionId
        ];
        $response = $offerSvc->get($params);
        $offerGroups = $response->data;

        if (! $offerGroups) {
            $message = sprintf("Unable to find any offers for promotion ID: %s. Please create an offer.", $promotionId);
            Flash::error($message);
            return redirect()->route('admin.offer.offers', $promotionId);
        }

        // ------------------------------
        // Get Various Dropdown Options
        // ------------------------------
        $participantTypes = $this->claimUtil->getParticipantTypes();
        $documentTypes    = $this->claimUtil->getDocumentTypes();
        $documentSuffixes = $this->claimUtil->getDocumentSuffixes();

        $listType = session()->get('claimListType', 'all');

        // ------------------------------
        // Show Page
        // ------------------------------
        $pageData = [
            'code' => $code,
            'mode' => 'new',
            'claimReviewUri' => '/admin/claim/review',
            'listType' => $listType,
            'claimFields' => $claimFields,
            'internalClaimFields' => $internalClaimFields,
            'formData' => $formData,
            'participantTypes' => $participantTypes,
            'offerGroups' => $offerGroups,
            'filesCode' => $filesCode,
            'uploadFiles' => $uploadFiles,
            'documentTypes' => $documentTypes,
            'documentSuffixes' => $documentSuffixes,
        ];

        return view('Claim::admin.edit', $pageData);
    }

    /**
     * Show the Review New Claim entry page.
     *
     * @return \Illuminate\Http\Response
     */
    public function review()
    {
        $errors = [];
        $formData = request()->all();
        $mode = $formData['mode'];

        // ---------------------------------
        // Save Form Data for re-display and Saving to Database
        // ---------------------------------
        $formData['class']              = 'claim';
        $formData['claim_items']        = request('claim_items', []);
        $formData['claim_participants'] = request('claim_participants', []);
        $formData['upload_files']       = request('upload_files', []);

        $dynamicFields = $formData['dynamicFields'];
        $claim         = $dynamicFields['claim'];
        $claimItems    = $formData['claim_items'];
        $uploadFiles   = $formData['upload_files'];
        $promotionId   = $formData['offer_promotion_id'];

        foreach ($formData as $key => $value) {
            if (isset($claim[$key]) && ! empty($claim[$key])) {
                $formData[$key] = $claim[$key];
            }
        }

        if ($mode == 'new') {
            $claimParticipants = $formData['claim_participants'];
        } else {
            $claimParticipant = $dynamicFields['claim_participant'];
            $formData['claim_participant'] = $claimParticipant;
        }

        unset($formData['dynamicFields']);

        $code = request('code', uniqid(''));
        session()->put($code, serialize($formData));

        if ($mode == 'new') {
            $returnURI = '/admin/claim/new/' . $promotionId . '/' . $code;
        } else {
            $returnURI = '/admin/claim/edit/' . $formData['claim_id'] . '/' . $claimParticipant['id'] . '/' . $code;
        }

        // ---------------------------------
        // Get Claim Submitter Info
        // ---------------------------------
        $submitter = $this->claimUtil->getSubmitterParticipant($errors);

        if (! $submitter) {
            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Get Form Fields
        // ------------------------------
        $claimFormSettings = FormEntity::findByKey(self::$claimFieldsExternalKey);

        if (! $claimFormSettings) {
            $errors[] = sprintf("Missing Form settings for Key = '%s'\n", self::$claimFieldsExternalKey);
            return redirect($returnURI)->with('errors', $errors);
        }

        $internalClaimFormSettings = FormEntity::findByKey(self::$claimFieldsInternalKey);

        if (! $internalClaimFormSettings) {
            $errors[] = sprintf("Missing Internal Form settings for Key = '%s'\n", self::$claimFieldsInternalKey);

            return redirect($returnURI)->with('errors', $errors);
        }

        // ---------------------------------
        // Validate Claim data
        // ---------------------------------
        $errors = Form::validateFormData('claim', $formData, $claimFormSettings);

        // ---------------------------------
        // Validate Claim Participant data
        // ---------------------------------
        if (empty($errors)) {
            // Check if we are validating a new Claim with multiple Participants
            if ($mode == 'new') {
                $submitterEmail = $submitter['email'];
                $emails         = [];
                $lineCount      = 0;
                foreach ($claimParticipants as $participant) {
                    ++$lineCount;
                    $participantEmail = $participant['participant_email'];
                    if ($participantEmail == $submitterEmail) {
                        $errors[] = sprintf(
                            "You cannot add yourself as an Additional Participant on line %s.\n",
                            $lineCount
                        );
                    }
                    if (! empty($emails[$participantEmail])) {
                        $errors[] = sprintf(
                            "You cannot have multiple of the same Additional Participant %s on line %s.\n",
                            $lineCount
                        );
                    }
                }
            } // Assume this is an update with only 1 Participant
            else {
                $errors = Form::validateFormData('claim_participant', $claimParticipant, $internalClaimFormSettings);
            }
        }

        if ((bool) setting('claim.use_simplified_lineitems') || (bool) setting('claim.offers.use_checkboxes')) {
            $formData = $this->transformSimplifiedLineItems($formData);
            $claimItems = $formData['claim_items'];
        }

        // Validate Claim Line Item data
        $maxLineItems = (int) setting('claim.max_lineitems', -1);
        if (empty($errors)) {
            if ($maxLineItems > -1 && count($claimItems) < 1) {
                $errors[] = sprintf("You must enter at least 1 Claim Line Item.\n");
            }
        }

        if (empty($errors)) {
            if (count($claimItems) > 0) {
                $quantityMin = setting('claim.lineitems.min_quantity', 1);
                $quantityMax = setting('claim.lineitems.max_quantity', 1000);
                $errors      = Form::validateFormData('claim_lineitem', $claimItems, $claimFormSettings);
                // Eoghan: The following code should be refactored but for now
                // it will suffice. Eventually, we should handle form data with
                // multiple rows in a more dynamic, UI configured way.
                $validator = Validator::make(['claim_item' => $claimItems], [
                    'claim_item.*.offer_group' => 'required',
                    'claim_item.*.name' => 'required',
                    'claim_item.*.qty' => 'required|numeric|between:' . $quantityMin . ',' . $quantityMax
                ]);

                if ($validator->fails()) {
                    foreach ($validator->errors()->all() as $error) {
                        $errors[] = preg_replace_callback('/claim_item.([0-9]+).qty\ must be/', function ($match) {
                            $index = (int)$match[0] + 1;

                            return "Claim item at position $index must have a quantity value";
                        }, $error);
                    }
                }
            }
        }

        if (! empty($errors)) {
            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Populate the Form Fields with the Form Data
        // ------------------------------
        $review      = false;
        $claimFields = Form::mergeFormFieldData('claim', $formData, $claimFormSettings, $review, $errors);

        if (! $claimFields) {
            if (empty($claimFields)) {
                $message = sprintf("The list of External Dynamic Claim Fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the External Dynamic Claim Field data.");
            }
            array_unshift($errors, $message);

            return redirect($returnURI)->with('errors', $errors);
        }

        $internalClaimFields = Form::mergeFormFieldData(
            'claim_participant',
            $formData,
            $internalClaimFormSettings,
            $review,
            $errors
        );

        if (! $internalClaimFields) {
            if (empty($internalClaimFields)) {
                $message = sprintf("The list of Internal Dynamic Claim Fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the Internal Dynamic Claim Field data.");
            }
            array_unshift($errors, $message);

            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Get Various Dropdown Options
        // ------------------------------
        $documentTypes    = $this->claimUtil->getDocumentTypes();
        $documentSuffixes = $this->claimUtil->getDocumentSuffixes();

        // ---------------------------------
        // Re-display Data
        // ---------------------------------
        $pageData = [
            'mode' => $mode,
            'code' => $formData['code'],
            'claimStoreUri' => '/admin/claim/store',
            'formData' => $formData,
            'claimFields' => $claimFields,
            'documentTypes' => $documentTypes,
            'documentSuffixes' => $documentSuffixes,
            'uploadFiles' => $uploadFiles,
            'internalClaimFields' => $internalClaimFields,
        ];

        return view('Claim::admin.review', $pageData);
    }

    /**
     * Store a Claim.
     *
     * @return \Illuminate\Http\Response
     */
    public function store()
    {
        $data = request()->all();

        $code        = $data['code'];
        $mode        = $data['mode'];
        $promotionId = $data['offer_promotion_id'];
        $claimId     = $data['claim_id'];

        // -----------------------------------------
        // Get Form Data saved in session by code key
        // -----------------------------------------
        $formData = unserialize(session()->get($code));
        if (! $formData) {
            return $this->claimUtil->reEnterClaim('/admin/claim/new');
        }

        if ($mode != 'new') {
            $claimParticipant = $formData['claim_participant'];
        }

        if ($mode == 'new') {
            $returnURI = '/admin/claim/new/' . $promotionId . '/' . $code;
        } else {
            $returnURI = '/admin/claim/edit/' . $claimId . '/' . $claimParticipant['id'] . '/' . $code;
        }

        // Check if the Back Button was pressed
        if (isset($data['backButton'])) {
            return redirect($returnURI)->with('formData', $formData);
        }

        if ((bool) setting('claim.use_simplified_lineitems') || (bool) setting('claim.offers.use_checkboxes')) {
            $formData = $this->transformSimplifiedLineItems($formData);
        }

        // -----------------------------------------
        // Create or Update Claim
        // -----------------------------------------
        $formData['class'] = 'claim';

        if ($mode == 'new') {
            $formData['action'] = 'createClaim';
        } else {
            $formData['action'] = 'updateClaim';
        }

        $response = $this->claimRepository->put($formData);

        if (! $response->result) {
            return redirect($returnURI)->with('errors', $response->errors);
        }

        // -----------------------------------------
        // Clear Session data for this Claim
        // -----------------------------------------
        session()->forget($code);

        $listType = ($mode == 'new') ? 'pending' : session()->get('claimListType', 'all');

        if (auth()->user()->can('claim.all.browse') || auth()->user()->can("claim.{$listType}.browse")) {
            $returnURI = '/admin/claim/list/' . $listType;
        } elseif (auth()->user()->can('claim.process.view')) {
            $returnURI = '/admin/claim/process';
        } elseif (auth()->user()->can('claim.issued.view')) {
            $returnURI = '/admin/claim/issued';
        } else {
            $returnURI = '/';
        }

        return redirect($returnURI);
    }

    /**
     * Show the View Claim page.
     *
     * @param  int $claimId
     * @param  int $claimParticipantId
     * @return \Illuminate\Http\Response
     * @throws EntryNotFoundException
     */
    public function show($claimId, $claimParticipantId)
    {
        $errors = [];

        $formData = request()->all();

        $listType  = session()->get('claimListType', 'all');
        $returnURI = '/admin/claim/list/' . $listType;

        // ---------------------------------
        // Retrieve Claim Data
        // ---------------------------------
        $params = [
            'id' => $claimId,
            'class' => 'claim',
            'action' => 'find',
            'mode' => 'array',
            'with' => [
                'lineItems',
                'claim_participants',
                'claim_participants.participant',
                'documents'
            ]
        ];

        $response = $this->claimRepository->get($params);

        if (! $response->result) {
            $errors = $response->errors;

            return redirect($returnURI)->with('errors', $errors);
        }

        $formData = $response->data;

        // ---------------------------------
        // Get Specific Claim Participant
        // ---------------------------------
        $claimParticipant = false;
        foreach ($formData['claim_participants'] as $participant) {
            if ((int) $participant['id'] === (int) $claimParticipantId) {
                $claimParticipant = $participant;
            }
        }

        if (! $claimParticipant) {
            $errors[] = sprintf("Could not find Claim Participant %s for Claim %s.", $claimParticipantId, $claimId);
            return redirect($returnURI)->with('errors', $errors);
        }

        $formData['claim_items']       = $formData['line_items'];
        $formData['claim_participant'] = $claimParticipant;

        unset($formData['line_items']);
        unset($formData['claim_participants']);

        // ------------------------------
        // Get Form Fields
        // ------------------------------
        $claimFormSettings = FormEntity::findByKey(self::$claimFieldsExternalKey);

        if (! $claimFormSettings) {
            $errors[] = sprintf("Missing Form settings for Key = '%s'\n", self::$claimFieldsExternalKey);

            return redirect($returnURI)->with('errors', $errors);
        }

        $internalClaimFormSettings = FormEntity::findByKey(self::$claimFieldsInternalKey);

        if (! $internalClaimFormSettings) {
            $errors[] = sprintf("Missing Internal Form settings for Key = '%s'\n", self::$claimFieldsInternalKey);

            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Populate the Form Fields with the Form Data
        // ------------------------------
        $review      = false;
        $claimFields = Form::mergeFormFieldData('claim', $formData, $claimFormSettings, $review, $errors);

        if (! $claimFields) {
            if (empty($claimFields)) {
                $message = sprintf("The list of External Dynamic Claim Fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the External Dynamic Claim Field data.");
            }
            array_unshift($errors, $message);

            return redirect($returnURI)->with('errors', $errors);
        }

        $internalClaimFields = Form::mergeFormFieldData(
            'claim_participant',
            $formData['claim_participant'],
            $internalClaimFormSettings,
            $review,
            $errors
        );

        if (! $internalClaimFields) {
            if (empty($internalClaimFields)) {
                $message = sprintf("The list of Internal Dynamic Claim Fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the Internal Dynamic Claim Field data.");
            }
            array_unshift($errors, $message);

            return redirect($returnURI)->with('errors', $errors);
        }

        // Grab the audit history of the claim and associated entities
        $histories = Audit::with('user')
                          ->where('context_id', $claimId)
                          ->where('context_type', Claim::class)
                          ->get();

        $notes = Note::with(['addedBy'])
                     ->where('table_primary_id', '=', $claimId)
                     ->where('table_name', '=', 'claim')
                     ->get()
                     ->toArray();

        // ------------------------------
        // Get Rule Log
        // ------------------------------
        $ruleSvc = app(\Ignite\Claim\Repositories\RuleRepository::class);

        $ruleLog = false;
        $params  = [
            'class' => 'claim_rule_log',
            'action' => 'getList',
            'mode' => 'array',
            'claim_participant_id' => $claimParticipantId
        ];

        $response = $ruleSvc->get($params);

        if ($response->result && is_array($response->data)) {
            $ruleLog = $response->data[0];
        }

        // ------------------------------
        // Get Various Dropdown Options
        // ------------------------------
        $documentTypes    = $this->claimUtil->getDocumentTypes();
        $documentSuffixes = $this->claimUtil->getDocumentSuffixes();

        // ---------------------------------
        // Display Data
        // ---------------------------------
        $pageData = [
            'claimParticipantId' => $claimParticipantId,
            'listType' => $listType,
            'documentTypes' => $documentTypes,
            'documentSuffixes' => $documentSuffixes,
            'formData' => $formData,
            'claimFields' => $claimFields,
            'internalClaimFields' => $internalClaimFields,
            'histories' => $histories,
            'notes' => $notes,
            'ruleLog' => $ruleLog,
        ];

        return view('Claim::admin.show', $pageData);
    }

    /**
     * Show the Edit Claim page.
     *
     * @param  int $claimId
     * @param  int $claimParticipantId
     * @param  string $code
     * @return \Illuminate\Http\Response
     * @throws EntryNotFoundException
     */
    public function edit($claimId, $claimParticipantId, $code = '')
    {
        $errors = [];

        $listType = session()->get('claimListType', 'all');

        $returnURI = '/admin/claim/list/' . $listType;

        // ---------------------------------
        // Check if First Display or Redisplay with Errors
        // ---------------------------------
        if ($code != '') {
            // ---------------------------------
            // Retrieve Claim Data from Session
            // ---------------------------------
            $formData = unserialize(session()->get($code));

            if (! isset($formData['maxLineitems'])) {
                $formData['maxLineitems'] = setting('claim.max_lineitems');
            }

            if (! isset($formData['maxParticipants'])) {
                $formData['maxParticipants'] = setting('claim.max_participants');
            }

            if (! $formData) {
                return $this->claimUtil->reEnterClaim($returnURI);
            }

            // ---------------------------------
            // Admin Edit Claim only has a Single Claim Participant, so we have to get the others for the Dropdown
            // ---------------------------------
            $params = [
                'class' => 'claim_participant',
                'action' => 'getList',
                'mode' => 'array',
                'claim_id' => $claimId
            ];

            $response = $this->claimRepository->get($params);

            $claimParticipants = (! empty($response->data)) ? $response->data : [];
        } else {
            $code = uniqid('');

            // ---------------------------------
            // Get Claim Data from Database
            // ---------------------------------
            $params = [
                'class' => 'claim',
                'action' => 'find',
                'mode' => 'array',
                'with' => ['lineItems', 'claim_participants', 'claim_participants.participant', 'documents'],
                'id' => $claimId
            ];

            $response = $this->claimRepository->get($params);

            if (! $response->result) {
                $errors = $response->errors;

                return redirect($returnURI)->with('errors', $errors);
            }

            $formData = $response->data;

            // ---------------------------------
            // Get Specific Claim Participant
            // ---------------------------------
            $claimParticipants = $formData['claim_participants'];

            $claimParticipant = false;


            foreach ($formData['claim_participants'] as $participant) {
                if ($participant['id'] == $claimParticipantId) {
                    $claimParticipant = $participant;
                }
            }

            if (! $claimParticipant) {
                $errors[] = sprintf("Could not find Claim Participant %s for Claim %s.", $claimParticipantId, $claimId);

                return redirect($returnURI)->with('errors', $errors);
            }

            $formData['claim_id']          = $formData['id'];
            $formData['claim_items']       = $formData['line_items'];
            $formData['claim_participant'] = $claimParticipant;
            $formData['upload_files']      = $this->claimUtil->convertDocumentsToUploadFiles($formData['documents']);

            unset($formData['line_items']);
            unset($formData['claim_participants']);
            unset($formData['documents']);
        }

        $filesCode   = $code;
        $uploadFiles = $formData['upload_files'];

        $promotionId = $formData['offer_promotion_id'];

        // ------------------------------
        // Get Form Fields
        // ------------------------------
        $claimFormSettings = FormEntity::findByKey(self::$claimFieldsExternalKey);

        if (! $claimFormSettings) {
            $errors[] = sprintf("Missing Form settings for Key = '%s'\n", self::$claimFieldsExternalKey);

            return redirect($returnURI)->with('errors', $errors);
        }

        $internalClaimFormSettings = FormEntity::findByKey(self::$claimFieldsInternalKey);

        if (! $internalClaimFormSettings) {
            $errors[] = sprintf("Missing Internal Form settings for Key = '%s'\n", self::$claimFieldsInternalKey);

            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Populate the Form Fields with the Form Data
        // ------------------------------
        $review      = false;
        $claimFields = Form::mergeFormFieldData('claim', $formData, $claimFormSettings, $review, $errors);

        if (! $claimFields) {
            if (empty($claimFields)) {
                $message = sprintf("The list of External Dynamic Claim Fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the External Dynamic Claim Field data.");
            }
            array_unshift($errors, $message);

            return redirect($returnURI)->with('errors', $errors);
        }

        $internalClaimFields = Form::mergeFormFieldData(
            'claim_participant',
            $formData['claim_participant'],
            $internalClaimFormSettings,
            $review,
            $errors
        );

        if (! $internalClaimFields) {
            if (empty($internalClaimFields)) {
                $message = sprintf("The list of Internal Dynamic Claim Fields was empty.");
            } else {
                $message = sprintf("There was a problem with merging the Internal Dynamic Claim Field data.");
            }
            array_unshift($errors, $message);

            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Get Offer Groups/Offers
        // ------------------------------
        $offerSvc      = app(OfferRepository::class);
        $params        = [
            'class' => 'claim_offer',
            'action' => 'getOfferGroups',
            'mode' => 'array',
            'status' => 1,
            'promotion_id' => $promotionId
        ];
        $response      = $offerSvc->get($params);
        $offerGroups = $response->data;

        if (! $offerGroups) {
            $errors[] = sprintf("Can't find Offers for Promotion Id = '%s'\n", $promotionId);
            $errors   = array_merge($errors, $response->errors);

            return redirect($returnURI)->with('errors', $errors);
        }

        // ------------------------------
        // Get data for Dropdowns, etc.
        // ------------------------------
        $participantTypes = $this->claimUtil->getParticipantTypes();
        $documentTypes    = $this->claimUtil->getDocumentTypes();
        $documentSuffixes = $this->claimUtil->getDocumentSuffixes();

        $participantOptions = [];
        foreach ($claimParticipants as $tmpParticipant) {
            $participantOptions[$tmpParticipant['id']] = $tmpParticipant['participant_email'];
            if (! empty($tmpParticipant['first'])) {
                $participantOptions[$tmpParticipant['id']] .= ' - ' . $tmpParticipant['first'] . ' ' . $tmpParticipant['last'];
            }
        }

        // ---------------------------------
        // Re-display Data
        // ---------------------------------
        $pageData = [
            'code' => $code,
            'filesCode' => $code,
            'mode' => 'edit',
            'claimReviewUri' => '/admin/claim/review',
            'listType' => $listType,
            'promotionId' => $promotionId,
            'formData' => $formData,
            'claimFields' => $claimFields,
            'internalClaimFields' => $internalClaimFields,
            'participantTypes' => $participantTypes,
            'participantOptions' => $participantOptions,
            'offerGroups' => $offerGroups,
            'documentTypes' => $documentTypes,
            'documentSuffixes' => $documentSuffixes,
            'uploadFiles' => $uploadFiles,
        ];

        return view('Claim::admin.edit', $pageData);
    }

    /**
     * Delete a Claim.
     *
     * @param  int  $claimId
     * @param  bool $claimParticipantId
     * @return \Illuminate\Http\Response
     */
    public function deleteClaim($claimId, $claimParticipantId = false)
    {
        $listType = session()->get('claimListType', 'all');
        $returnURI = '/admin/claim/list/' . $listType;

        // Delete Claim
        $response = $this->claimRepository->put([
            'class' => 'claim',
            'action' => 'deleteClaim',
            'claim_id' => $claimId,
            'claim_participant_id' => $claimParticipantId
        ]);

        $result = $response->result;
        $errors = $response->errors;


        if ($result) {
            if ($claimParticipantId) {
                $message = sprintf("Claim Id %s, Claim Participant Id %s, deleted.", $claimId, $claimParticipantId);
            } else {
                $message = sprintf("Claim Id %s, deleted.", $claimId);
            }
            $status = 'success';
        } else {
            $message = $errors;
            $status = 'error';
        }

        if (request()->wantsJson()) {
            return response()->json(compact('status', 'message'), ($result) ? 200 : 422);
        }

        if ($result) {
            $this->flash('success', $message);
            return redirect($returnURI);
        }

        $this->flash('danger', $message);
        return redirect($returnURI);
    }

    /**
     * Add a Claim Note.
     *
     * @return \Illuminate\Http\Response
     */
    public function addNote()
    {
        $data = request()->all();

        $rules = [
            'user_id' => 'required|integer|min:1',
            'add_note' => 'required|string|min:10',
        ];

        $userId    = $data['user_id'];
        $claimId   = $data['claim_id'];
        $note      = $data['add_note'];
        $errors    = [];

        $this->validate(request(), $rules);

        try {
            Note::forceCreate([
                'by_user_id'       => auth()->user()->getKey(),
                'login_user_id'    => app('impersonation')->getImpersonatorId() ?? auth()->user()->getKey(),
                'table_name'       => 'claim',
                'table_primary_id' => $claimId,
                'message'          => $note,
            ]);
        } catch (Exception $e) {
            $errors[] = "Unable to Create Note for Claim Id $claimId.";
            $errors[] = $e->getMessage();

            return redirect()->back()->with('errors', $errors);
        }

        $this->flash('success', 'Note added successfully.');
        return redirect("/admin/claim/view/$claimId/$userId");
    }

    /**
     * Test if Claim would be Approved by Promotion Caps.
     *
     * @param  int $claimId
     * @param  int $claimParticipantId
     * @return \Illuminate\Http\Response
     */
    public function testApprove($claimId, $claimParticipantId)
    {
        $returnURI = '/admin/claim/view/' . $claimId . '/' . $claimParticipantId;

        // Check if Claim would be Approved by any Promotion Caps that are Active
        // TODO: Resolve the cap check out of the IoC container
        $capCheck = app(CapCheck::class, ['claimRepository' => $this->claimRepository]);
        //$capCheck = new CapCheck($this->claimRepository, new OfferRepository(), new TransactionRepository());
        $response = $capCheck->checkCap($claimParticipantId);

        if (! $response->result) {
            $this->flash('error', $response->errors);
            return redirect($returnURI);
        }

        $message = sprintf(
            "Claim Id %s, for Participant Id %s, would be Approved.",
            $claimId,
            $claimParticipantId
        );

        $this->flash('status', $message);
        return redirect($returnURI);
    }

    /**
     * In the case of a simplified UI for adding line items, we need transform the
     * lineitems for use in the Rule engine.
     *
     * @param  array $formData
     * @return array
     */
    private function transformSimplifiedLineItems($formData)
    {
        if (! isset($formData['claim_items']) || ! isset($formData['claim_items'][0])) {
            return $formData;
        }

        $firstItem = $formData['claim_items'][0];
        if (strpos($firstItem['offer_id'], ',') !== false && strpos($firstItem['name'], '|') !== false) {
            $items = array_combine(
                preg_split('/\s?,\s?/', $firstItem['offer_id']),
                preg_split('/\s?\|\s?/', $firstItem['name'])
            );
            $lineItems = [];
            foreach ($items as $id => $name) {
                $lineItem = $firstItem;
                $lineItem['name'] = $name;
                $lineItem['offer_id'] = $id;
                $lineItems[] = $lineItem;
            }
            $formData['claim_items'] = $lineItems;
        }

        return $formData;
    }
}
