<?php

namespace Ignite\Claim\Entities;

use Ignite\Claim\Models\Audit\ClaimDriver;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\Transaction;
use Ignite\Core\Entities\User;
use Illuminate\Database\Eloquent\SoftDeletes;
use Ignite\Claim\Entities\Dynamic\DynamicModel;
use Ignite\Claim\Traits\ValidatesFields;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

/**
 * Class ClaimParticipant
 *
 * @property Claim $claim
 * @property ClaimLineitem $lineitems
 * @property Transaction $transactions
 * @property RuleLog $rule_log
 * @property Participant $participant
 * @property User $user
 */
class ClaimParticipant extends DynamicModel implements AuditableContract
{
    use Auditable, ValidatesFields, SoftDeletes;

    const TABLE_KEY = 'table.columns.claim_participant';

    const STATUS_WAITING = 'waiting';
    const STATUS_PENDING = 'pending';
    const STATUS_DECLINED = 'declined';
    const STATUS_APPROVED = 'approved';
    const STATUS_CANCELLED = 'cancelled';
    const STATUS_ISSUED = 'issued';

    /**
     * The name of the database table.
     * @var string
     */
    protected $table = 'claim_participant';

    /**
     * The relations to eager load on every query.
     *
     * @var array
     */
    protected $with = ['claim', 'lineitems'];

    /**
     * The name of the primary key.
     * @var string
     */
    protected $primaryKey = 'id';

    /**
     * Should the timestamps be audited?
     *
     * @var bool
     */
    protected $auditTimestamps = false;

    /**
     * Custom Audit Driver
     *
     * @var ClaimDriver
     */
    protected $auditDriver = ClaimDriver::class;

    /**
     * Attributes to exclude from the Audit.
     *
     * @var array
     */
    protected $auditExclude = [
        'id',
        'updated_at',
        'created_at'
    ];

    /**
     * The list of dynamic fields.
     * @var array
     */
    public static $fields = [
        'id'                => ['type' => 'integer'],
        'claim_id'          => ['type' => 'integer', 'validate' => 'required'],
        'user_id'           => ['type' => 'integer', 'validate' => 'required'],
        'participant_type'  => ['type' => 'string', 'validate' => 'required'],
        'participant_email' => ['type' => 'string', 'validate' => 'required'],
        'activity_date'     => ['type' => 'date'],
        'date_submitted'    => ['type' => 'date'],
        'date_approved'     => ['type' => 'date'],
        'date_canceled'     => ['type' => 'date'],
        'date_declined'     => ['type' => 'date'],
        'date_issued'       => ['type' => 'date'],
        'value'             => ['type' => 'double', 'default' => '0'],
        'value_adjust'      => ['type' => 'double', 'default' => '0'],
        'value_calculated'  => ['type' => 'double', 'default' => '0'],
        'reason_declined'   => ['type' => 'string'],
        'internal_notes'    => ['type' => 'string'],
        'external_notes'    => ['type' => 'string'],
        'status'            => ['type' => 'string', 'default' => 'pending'],
        'archived'          => ['type' => 'integer', 'default' => '0'],
    ];

    /** @var array */
    public static $statuses = [
        'approved' => 'approved',
        'cancelled' => 'cancelled',
        'declined' => 'declined',
        'issued'   => 'issued',
        'pending'  => 'pending',
        'waiting'  => 'waiting',
    ];

    /**
     * The list of acceptable decline reasons.
     * @var array
     */
    public static $declinedReasons = [
        ''            => '--- Select Reason ---',
        'custom'      => 'Custom Reason.',
        'revenue'     => 'Inconsistent Revenue Value.',
        'support_doc' => 'Error in Support Documentation.',
        'offer_dates' => 'Offer Not Active when submitted.',
    ];

    /**
     * The relationship to the claim table.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function claim()
    {
        return $this->belongsTo(Claim::class, 'claim_id', 'id');
    }

    /**
     * The relationship to the claim line items.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function lineitems()
    {
        return $this->hasMany(ClaimLineitem::class, 'claim_id', 'claim_id');
    }

    /**
     * The relationship to the transactions for the claim participant.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function transactions()
    {
        return $this->hasMany(Transaction::class, 'user_id', 'user_id');
    }

    /**
     * The relationship to the rule log for the claim participant.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function rule_log()
    {
        return $this->hasOne(RuleLog::class, 'claim_participant_id', 'id');
    }

    /**
     * The parallel participant record relationship.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function participant()
    {
        return $this->belongsTo(Participant::class, 'user_id', 'user_id');
    }

    /**
     * The parallel user record relationship.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'user_id');
    }

    /**
     * The identifier for the record in the audit log.
     *
     * @return mixed
     */
    public static function getAuditFriendlyField()
    {
        return 'participant_email';
    }
}
