<?php

namespace Ignite\Claim\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class File extends Model implements AuditableContract
{
    use Auditable;

    /**
     * The custom table name.
     * @var string
     */
    protected $table = 'claim_file';

    /**
     * The custom primary key.
     * @var string
     */
    protected $primaryKey  = 'id';

    /**
     * Custom Audit Driver
     *
     * @var \Ignite\Claim\Models\Audit\ClaimDriver
     */
    protected $auditDriver = \Ignite\Claim\Models\Audit\ClaimDriver::class;

    /**
     * Gets a File record using an Obfuscated File Id
     *
     * @todo Two unused variables below.
     *
     * @param  string $fileIdentifier
     * @return File
     * @throws \Exception
     */
    public static function getObfuscated($fileIdentifier)
    {
        // Decode the File Identifier to get the File Id
        $data = scramble()->Unscramble($fileIdentifier);
        $parts = explode('-', $data);
        $partsCount = count($parts);

        // Check basic Validation
        if ($partsCount != 3) {
            throw new \Exception("File request validation failed.");
        }

        // Extract the Request Parameters
        $fileId         = $parts[$partsCount-1];
        $relatedId      = $parts[$partsCount-2];
        $relatedName    = $parts[$partsCount-3];

        // Try to find the Requested File record
        $file = static::find($fileId);

        if (! $file) {
            throw new \Exception("Requested File not found.");
        }

        // Check any additional validation
        if ($relatedId !== false && $file->related_id != $relatedId) {
            throw new \Exception("File id validation failed.");
        }

        if ($relatedName !== false && $file->related_name != $relatedName) {
            throw new \Exception("File name validation failed.");
        }

        return $file;
    }

    /**
     * Get the allowed/required document types.
     *
     * @todo: Move this to ClaimUtil::getDocumentTypes() since this is
     *        claim specific documents (claim.document.requires)
     *
     * @return array
     */
    public static function getDocumentTypes()
    {
        $documentTypes = [];

        $value = setting('claim.document.requires', '');

        $docTypes = preg_split('/[|\n]/', $value);

        foreach ($docTypes as $docType) {
            $parts = explode(':', $docType);
            $documentType = [];

            $documentType['name'] = $parts[0];
            $documentType['min']  = (isset($parts[1])) ? $parts[1] : '0';
            $documentType['max']  = (isset($parts[2])) ? $parts[2] : '10';

            if ('' != $documentType['name'] && $documentType['max'] > 0) {
                $documentTypes[] = $documentType;
            }
        }

        return $documentTypes;
    }

    /**
     * The identifier for the record in the audit log.
     *
     * @return mixed
     */
    public static function getAuditFriendlyField()
    {
        return 'original_name';
    }
}
