<?php

namespace Ignite\Claim\Models;

use Ignite\Claim\Events\ClaimEvent;
use Ignite\Core\Entities\Exception;
use Ignite\Claim\Emails\ClaimEmail;
use Illuminate\Support\Facades\Mail;
use Ignite\Core\Entities\Setting;

class ClaimEmailSender
{
    /**
     * Handle the email status.
     *
     * @param  ClaimEvent $event
     * @return bool
     * @throws Exception
     */
    public function handleEmailEvent(ClaimEvent $event)
    {
        if (empty($event->data)) {
            throw new Exception("Email could not be sent: The data field was not set for '{$event->eventType}' event.");
        }

        $settings = $this->getEmailSettings($event);
        if (! $settings['active']) {
            return false;
        }

        $data = $event->data;
        $data['email_settings'] = [
            'eventType'   => $event->eventType,
            'eventActive' => $settings['active'],
            'name'        => $this->getRecipientName($data),
            'toName'      => $this->getRecipientName($data),
            'toEmail'     => $this->getRecipientEmail($data),
            'fromName'    => config('mail.from.name'),
            'fromEmail'   => config('mail.from.address'),
            'subject'     => $this->parse($settings['subject'], $data),
            'template'    => $settings['template'],
        ];

        // Send Email
        return Mail::send(new ClaimEmail($data));
    }

    /**
     * @param  ClaimEvent $event
     * @return array
     * @throws Exception
     */
    protected function getEmailSettings(ClaimEvent $event)
    {
        // Get the settings for this email event.
        $key = 'claim.email.' . $event->eventType;
        $setting = setting($key);

        // Check if we have email settings available
        if (empty($setting)) {
            throw new Exception("Email could not be sent: No email settings available: $key");
        }

        $settings = json_decode($setting, true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new Exception('Unable to parse the email setting data: ' . json_last_error_msg());
        }

        if (empty($settings['template'])) {
            throw new Exception(
                sprintf("No email template configured for '%s' event, can't process email.", $event->eventType)
            );
        }

        if (empty($settings['subject'])) {;
            throw new Exception(sprintf("No subject configured for '%s' event, can't process email.", $event->eventType));
        }

        if (empty($settings['active'])) {
            $settings['active'] = false;
        }

        $settings['active'] = (bool) $settings['active'];

        return $settings;
    }

    /**
     * Parse and substitute variable values in a string.
     *
     * @param  string $string
     * @param  array  $data
     * @return string
     */
    protected function parse($string, $data)
    {
        $pattern = '~\{[A-z0-9_]+\}~';

        if (preg_match_all($pattern, $string, $matches)) {
            if (isset($matches[0]) && is_array($matches[0])) {
                foreach($matches[0] as $match) {
                    $string = $this->interpolate($string, $data, $match);
                }
            }
        }

        return $string;
    }

    /**
     * Interpolate the actual variable value into the provided string.
     *
     * @param  string $string
     * @param  array  $data
     * @param  string $match
     * @return string
     */
    protected function interpolate($string, $data, $match)
    {
        $match = trim($match, '{}');
        $value = null;
        $transformations = ['claim_id', 'name'];

        foreach ($transformations as $var) {
            $value = $this->applyTransformation($var, $data);
            $string = str_replace('{' . $match . '}', $value, $string);
        }

        return trim($string);
    }

    /**
     * Apply the transformation for the given variable.
     *
     * @param  string $variable
     * @param  array  $data
     * @return string
     */
    protected function applyTransformation($variable, $data)
    {
        $method = sprintf('transform%s', ucfirst(camel_case($variable)));

        if (! method_exists($this, $method)) {
            throw new \BadMethodCallException(
                "You attempted to transform a variable without creating a transform method: $method"
            );
        }

        return $this->$method($data);
    }

    /**
     * Transform the claim id variable.
     *
     * @param  array $data
     * @return string
     */
    protected function transformClaimId($data)
    {
        return $data['id'];
    }

    /**
     * Transform the participant name variable.
     *
     * @param  array $data
     * @return string
     */
    protected function transformName($data)
    {
        return $this->getRecipientName($data);
    }

    /**
     * Get the name of the recipient.
     *
     * @param array $data
     * @return string
     */
    protected function getRecipientName($data)
    {
        if (empty($data['email_participant']['participant'])) {
            return 'Rewards Program Participant';
        }

        return $data['email_participant']['participant']['first'] . ' ' . $data['email_participant']['participant']['last'];
    }

    /**
     * Get the email of the recipient.
     *
     * @param  array $data
     * @return string
     */
    protected function getRecipientEmail($data)
    {
        if (! empty($data['email_settings']['toEmail'])) {
            return $data['email_settings']['toEmail'];
        }

        return $data['email_participant']['participant_email'];
    }
}
