<?php

namespace Ignite\Claim\Http\Controllers;

use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Ignite\Claim\Repositories\ClaimRepository;
use Ignite\Claim\Repositories\OfferRepository;
use Ignite\Core\Repositories\ParticipantRepository;
use Ignite\Claim\Entities\Rule;
use Ignite\Claim\Entities\RuleCondition;
use Ignite\Claim\Repositories\RuleRepository;

class RuleAdminController extends Controller
{
    /** @var ClaimRepository $claimRepository */
    private $claimRepository;

    /** @var ParticipantRepository */
    private $participantRepository;

    /** @var OfferRepository */
    private $offerRepository;

    /** @var RuleRepository */
    private $ruleRepository;

    /**
     * RuleAdminController constructor.
     *
     * @param ClaimRepository $claimRepository
     * @param ParticipantRepository $participantRepository
     * @param OfferRepository $offerRepository
     * @param RuleRepository $ruleRepository
     */
    public function __construct(
        ClaimRepository $claimRepository,
        ParticipantRepository $participantRepository,
        OfferRepository $offerRepository,
        RuleRepository $ruleRepository
    ) {
        $this->middleware(['auth']);
        $this->claimRepository = $claimRepository;
        $this->participantRepository = $participantRepository;
        $this->offerRepository = $offerRepository;
        $this->ruleRepository = $ruleRepository;
    }

    /**
     * Display a listing of Promotions.
     *
     * @param int $campaignId
     * @return Response
     */
    public function index($campaignId = 1)
    {
        // Get available Campaigns
        $params = [
            'class' => 'claim_offer_campaign',
            'action' => 'getList',
            'mode' => 'array',
            'status' => '1'
        ];
        $response = $this->offerRepository->get($params);
        $campaigns = $response->data;

        // Get Promotions for Campaign
        $params = [
            'class' => 'claim_offer_promotion',
            'action' => 'getList',
            'mode' => 'array',
            'status' => '1',
            'campaign_id' => $campaignId,
            'orderby' => 'name'
        ];
        $response = $this->offerRepository->get($params);
        $promotions = $response->data;

        return view('Claim::admin.rules.index', compact('campaigns', 'campaignId', 'promotions'));
    }

    /**
     * Display a listing of Rule Sets for a Promotion
     *
     * @param int $promotionId
     * @return Response
     */
    public function listRules($promotionId)
    {
        $returnURI = '/admin/rule/promotions';

        // Get Promotion
        $response = $this->offerRepository->get([
            'class' => 'claim_offer_promotion',
            'action' => 'find',
            'mode' => 'array',
            'id' => $promotionId,
            'status' => '1'
        ]);
        $promotion = $response->data;

        if (! $promotion) {
            $errors[] = sprintf("Promotion Id '%s' not found.", $promotionId);
            $errors = array_merge($errors, $response->errors);
            return redirect($returnURI)->with('errors', $errors);
        }

        // Get Rules for Promotion
        $response = $this->ruleRepository->get([
            'class' => 'claim_rule',
            'action' => 'getList',
            'mode' => 'array',
            'with' => ['participants', 'offers', 'values', 'conditions'],
            'offer_promotion_id' => $promotionId,
            'orderBy' => ['order']
        ]);
        $rules = $response->data;

        // Get Claims for Promotion
        $response = $this->claimRepository->get([
            'class' => 'claim',
            'action' => 'getList',
            'mode' => 'array',
            'with' => ['user'],
            'offer_promotion_id' => $promotionId,
            'orderBy' => ['id|desc']
        ]);
        $claims = $response->data;

        // Build Select Options
        $allRulesSelect = [];
        foreach ($rules as $rule) {
            $tmpRule = [];
            $tmpRule['id']      = $rule['id'];
            $tmpRule['name']    = $rule['name'];
            $tmpRule['status']  = $rule['status'];
            $allRulesSelect[] = $tmpRule;
        }

        $claimsSelect = [];
        foreach ($claims as $claim) {
            $tmpClaim = [];
            $tmpClaim['id']         = $claim['id'];
            $tmpClaim['first']      = $claim['user']['first'];
            $tmpClaim['last']       = $claim['user']['last'];
            $tmpClaim['claim_date'] = $claim['activity_date'];
            $claimsSelect[] = $tmpClaim;
        }

        // Build has / not has lists for Participant Left Right Selectors
        $participantTypes   = $this->participantRepository->getTypes();
        $pageData = [
            'promotionId'       => $promotionId,
            'rules'             => $rules,
            'allRulesSelect'    => $allRulesSelect,
            'testRules'         => [],
            'claimsSelect'      => $claimsSelect,
            'participantTypes'  => $participantTypes,
        ];

        return view('Claim::admin.rules.ruleList', $pageData);
    }

    /**
     * Show Edit Rule
     *
     * @param  int    $ruleId
     * @param  string $promotionId
     * @param  string $code
     * @return Response
     */
    public function showEdit($ruleId, $promotionId = '', $code = '')
    {
        $formData = request()->all();

        $returnURI = '/admin/rule/promotions';

        // Get Rule
        if ($ruleId != 'new') {
            $response = $this->ruleRepository->get([
                'class' => 'claim_rule',
                'action' => 'find',
                'mode' => 'array',
                'with' => ['participants', 'offers', 'values', 'conditions'],
                'id' => $ruleId
            ]);
            $rule = $response->data;
            $mode = 'edit';
        } elseif ($code != '') {
            $formData = unserialize(session()->get($code));
            $rule = $formData;
            $mode = $formData['mode'];
        } else {
            $rule = $this->ruleRepository->toArray('claim_rule');
            $rule['id'] = 'new';
            $rule['offer_promotion_id'] = $promotionId;
            $rule['participants'] = [];
            $rule['values'] = [];
            $rule['offers'] = [];
            $rule['conditions'] = [];
            $mode = 'new';
        }

        // Get Promotion
        $promotionId = $rule['offer_promotion_id'];
        $response = $this->offerRepository->get([
            'class' => 'claim_offer_promotion',
            'action' => 'find',
            'mode' => 'array',
            'with' => ['offers'],
            'id' => $promotionId,
            'status' => '1'
        ]);
        $promotion = $response->data;

        if (! $promotion) {
            $errors[] = sprintf("Promotion Id '%s' not found.", $promotionId);
            $errors = array_merge($errors, $response->errors);
            return redirect($returnURI)->with('errors', $errors);
        }

        // Build has / not has lists for Offer Left Right Selectors
        $offersLookup = [];
        foreach ($promotion['offers'] as $offer) {
            $offersLookup[$offer['id']] = $offer;
        }

        $hasOffers = [];
        foreach ($rule['offers'] as $tmpOffer) {
            $offerId = $tmpOffer['offer_id'];
            $offer = $offersLookup[$offerId];
            $tmpOffer['name'] = $offer['name'];
            $tmpOffer['sku'] = $offer['sku'];
            $tmpOffer['offer_group'] = $offer['offer_group'];
            $hasOffers[$offerId] = $tmpOffer;
        }

        $notHasOffers = [];
        foreach ($offersLookup as $id => $tmpOffer) {
            if (! isset($hasOffers[$id])) {
                $offer = [];
                $offer['id'] = 'new';
                $offer['offer_id'] = $tmpOffer['id'];
                $offer['name'] = $tmpOffer['name'];
                $offer['sku'] = $tmpOffer['sku'];
                $offer['offer_group'] = $tmpOffer['offer_group'];
                $notHasOffers[$id] = $offer;
            }
        }

        // -------------------------------------
        // Build has / not has lists for Participant Left Right Selectors
        // -------------------------------------
        $participantTypes = $this->participantRepository->getTypes();

        $hasParticipants = [];
        foreach ($rule['participants'] as $participant) {
            $participantType = $participant['participant_type'];
            $hasParticipants[$participantType]  = $participant;
        }

        $notHasParticipants = [];
        foreach ($participantTypes as $participantType => $name) {
            if (! isset($hasParticipants[$participantType])) {
                $participant = [];
                $participant['id'] = 'new';
                $participant['participant_type'] = $participantType;
                $notHasParticipants[$participantType] = $participant;
            }
        }

        $pageData = [
            'code'                    => $code,
            'mode'                    => $mode,
            'rule'                    => $rule,
            'promotion'               => $promotion,
            'offersLookup'            => $offersLookup,
            'valueSourceOptions'      => Rule::$valueSourceOptions,
            'valueTypeOptions'        => Rule::$valueTypeOptions,
            'participantsModeOptions' => Rule::$selectModeOptions,
            'offersModeOptions'       => Rule::$selectModeOptions,
            'directiveOptions'        => RuleCondition::$directives,
            'participantTypes'        => $participantTypes,
            'hasParticipants'         => $hasParticipants,
            'notHasParticipants'      => $notHasParticipants,
            'hasOffers'               => $hasOffers,
            'notHasOffers'            => $notHasOffers,
            'dbColumns'               => RuleCondition::getComparableColumns(),
        ];

        return view('Claim::admin.rules.editRules', $pageData);
    }


    /**
     * Store Rule
     *
     * @return Response
     */
    public function storeRule()
    {
        $formData = request()->all();

        // -------------------------------------
        // Get Data from POST
        // -------------------------------------
        $formData['participants'] = (! empty($formData['participants'])) ? $formData['participants'] : [];
        $formData['offers']       = (! empty($formData['offers'])) ? $formData['offers'] : [];
        $formData['values']       = (! empty($formData['values'])) ? $formData['values'] : [];
        $formData['conditions']   = (! empty($formData['conditions'])) ? $formData['conditions'] : [];
        $formData['status']       = (! empty($formData['status'])) ? $formData['status'] : 0;

        $mode = $formData['mode'];
        $ruleId = (! empty($formData['id'])) ? $formData['id'] : '';
        $promotionId = (! empty($formData['offer_promotion_id'])) ? $formData['offer_promotion_id'] : '';

        // Save Form Data for re-display and Saving to Database
        $code = request('code', uniqid(''));
        /*if (empty($formData['code'])) {
            $code = uniqid('');
        }*/
        session()->put($code, serialize($formData));
        $returnURI = "/admin/rule/edit/$ruleId/$promotionId/$code";

        // Create or Update Rule
        $formData['class'] = 'claim_rule';
        $formData['action'] = ($mode == 'new') ? 'createRule' : 'updateRule';
        $response = $this->ruleRepository->put($formData);

        if (! $response->result) {
            flash()->success('There was an error while saving the rule')->important();
            return redirect($returnURI)->with('errors', $response->errors);
        }

        // Clear session data for this rule
        session()->forget($code);

        flash()->success('Rule saved successfully!')->important();

        return redirect('/admin/rule/rules/' . $promotionId);
    }

    /**
     * Delete a Rule
     *
     * @param  $ruleId
     * @param  string $promotionId
     * @return Response
     */
    public function deleteRule($ruleId, $promotionId = '')
    {
        $returnURI = $promotionId > 0 ? '/admin/rule/rules/' . $promotionId : '/admin/rule/promotions/';
        $errorURI = '/admin/rule/edit/' . $ruleId;

        // Delete Rule
        $response = $this->ruleRepository->put([
            'class' => 'claim_rule',
            'action' => 'deleteRule',
            'rule_id' => $ruleId
        ]);

        if ($response->result) {
            flash()->success(sprintf('Successfully deleted rule with ID %s.', $ruleId))->important();
            return redirect($returnURI);
        }

        return redirect($errorURI)->with('errors', $response->errors);
    }
}
