<?php

namespace Ignite\Claim\Models\Legacy;

use Illuminate\Support\Carbon;

class TypeNegotiator
{
    /** @var string */
    protected $value;

    /** @var TableConfig */
    protected $config;

    /**
     * TypeNegotiator.
     *
     * @param string $value
     * @param TableConfig $config
     */
    public function __construct($value, TableConfig $config)
    {
        $this->value = is_string($value) ? trim($value) : $value;
        $this->config = $config;
    }

    /**
     * Apply the type negotiation.
     *
     * @param  string $field
     * @return mixed
     */
    public function apply($field)
    {
        return call_user_func_array([$this, $this->config->get('type')], compact('field'));
    }

    /**
     * Handle a string type.
     *
     * @param  string $field
     * @return string
     */
    protected function string($field)
    {
        // Possibly transliterate, swap diacritics, if necessary.

        // Check if the value is an array, if so, implode it as comma-separated
        if (is_array($this->value)) {
            $this->value = implode(',', $this->value);
        }

        return $this->value;
    }

    /**
     * Handle a password type.
     *
     * @param  string $field
     * @return string
     */
    protected function password($field)
    {
        return $this->value;
    }

    /**
     * Handle a textarea type.
     *
     * @param  string $field
     * @return string
     */
    protected function textarea($field)
    {
        // Possibly transliterate, swap diacritics, if necessary.

        return $this->value;
    }

    /**
     * Handle a blob type.
     *
     * @param  string $field
     * @return string
     */
    protected function blob($field)
    {
        return $this->value;
    }

    /**
     * Handle a integer type.
     *
     * @param  string $field
     * @return int
     */
    protected function int($field)
    {
        return (int) $this->removeNonNumericDisallowedCharacters();
    }

    /**
     * Handle a double type.
     *
     * @param  string $field
     * @return float
     */
    protected function double($field)
    {
        return (float) $this->removeNonNumericDisallowedCharacters();
    }

    /**
     * Handle a date type.
     *
     * @param  string $field
     * @return string
     */
    protected function date($field)
    {
        try {
            return Carbon::createFromTimestamp($this->timestamp($field))->startOfDay()->format('Y-m-d');
        } catch (\Exception $e) {
            return date(strtotime($this->value, 'Y-m-d'));
        }
    }

    /**
     * Handle a datetime type.
     *
     * @param  string $field
     * @return string
     */
    protected function datetime($field)
    {
        try {
            return Carbon::createFromTimestamp($this->timestamp($field))->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            return date(strtotime($this->value, 'Y-m-d H:i:s'));
        }
    }

    /**
     * Handle a timestamp type.
     *
     * @param  string $field
     * @return int
     */
    protected function timestamp($field)
    {
        try {
            return Carbon::createFromTimestamp($this->value)->getTimestamp();
        } catch (\Exception $e) {
            return strtotime($this->value);
        }
    }

    /**
     * Remove all non numeric characters except for a period.
     *
     * @return string
     */
    protected function removeNonNumericDisallowedCharacters()
    {
        return preg_replace('/[^0-9.]/', '', $this->value);
    }
}
